#!/usr/bin/python3

# Copyright (c) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

#coding:utf-8
"""
Mod_lsapi hooks manager for Cpanel EA4
-i - installs hooks
-d - removes hooks
"""
import os
import sys
import syslog

import yaml


YAML_HOOK_BASE = "/var/cpanel/hooks.yaml"
MANAGE_HOOK = "/usr/local/cpanel/bin/manage_hooks"


def log_error(msg):
    """
    Write message to log
    """
    syslog.syslog("%s: %s" % (__file__, msg))


def is_hook_installed(category, event, hook_script_path):
    """
    Check is hook installed in current cpanel
    """
    try:
        stream = open(YAML_HOOK_BASE, "r")
    except (IOError, OSError):
        return False

    hook_base = yaml.load(stream)
    try:
        for i in hook_base[category][event]:
            if i["hook"] == hook_script_path:
                return True
    except KeyError:
        return False
    return False


def install_hook(hook_script_path, category, event, stage):
    """
    Install event hook for cpanel
    """
    if not is_hook_installed(category, event, hook_script_path):
        cmd = "%s add script %s --category=%s --event=%s --stage=%s --manual" \
              % (MANAGE_HOOK, hook_script_path, category, event, stage)

        if os.system(cmd):
            log_error("Can`t add hook to category:%s event:%s" % (category, event))
            return 1

    return 0


def remove_hook(hook_script_path, category, event, stage):
    """
    Remove cpanel event hook
    """
    cmd = "%s delete script %s --category=%s --event=%s --stage=%s --manual" \
          % (MANAGE_HOOK, hook_script_path, category, event, stage)

    if os.system(cmd):
        log_error("Can't delete hook to category:%s event:%s" % (category, event))
        return 1

    return 0


def main(argv):
    """
    Install or delete cpanel event hook
    """
    result = 0
    if "-i" in argv:
        # install hooks
        result = install_hook("/usr/share/lve/modlscapi/hooks/cpanel_post_upcp_lsapi_hook.py",
                              "System", "upcp", "post") or result
        result = install_hook("/usr/share/lve/modlscapi/hooks/cpanel_post_upcp_check_handlers_hook.py",
                              "System", "upcp", "post") or result
        result = install_hook("/usr/share/lve/modlscapi/hooks/cpanel_pre_upcp_check_handlers_hook.py",
                              "System", "upcp", "pre") or result

    elif "-d" in argv:
        # remove hooks
        result = remove_hook("/usr/share/lve/modlscapi/hooks/cpanel_post_upcp_lsapi_hook.py",
                             "System", "upcp", "post") or result
        result = remove_hook("/usr/share/lve/modlscapi/hooks/cpanel_post_upcp_check_handlers_hook.py",
                             "System", "upcp", "post") or result
        result = remove_hook("/usr/share/lve/modlscapi/hooks/cpanel_pre_upcp_check_handlers_hook.py",
                             "System", "upcp", "pre") or result

    return result


if "__main__" == __name__:
    sys.exit(main(sys.argv[1:]))


