/*
 * Copyright (c) Cloud Linux GmbH & Cloud Linux Software, Inc 2010-2018 All Rights Reserved
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * The part the support library for lsapi & proxy_lsapi Apache modules
 * author Alexander Demeshko <ademeshko@cloudlinux.com>
 *
 * Here you define exportable functions for libscapi clients.
 */

#ifndef _LSCAPI_H_
#define _LSCAPI_H_

#define LSCAPI_PATH_PREFIX "lsapi_"


#include <sys/uio.h>
#include <sys/types.h>
#include <stdbool.h>
#include <stdint.h>

#include <lsapidef.h>
#include <lscapi_config.h>

#include <liblsapi-sha1.h>

#define LSCAPI_SECRET_SIZE 16
#define LSCAPI_SALT_SIZE 16
#define LSCAPI_SHA1_SIGN_SIZE (LSCAPI_SALT_SIZE + SHA1_DIGEST_SIZE)
#define LSCAPI_ENCODED_SHA1_SIGN_SIZE (2*LSCAPI_SHA1_SIGN_SIZE + 1)
#define LSCAPI_ENCODED_SHA1_SIGN_WITHOUT_SALT_SIZE (2*SHA1_DIGEST_SIZE + 1)


// lsapi structs
typedef struct lsapi_req_header lsapi_req_header_t;
typedef struct lsapi_http_header_index lsapi_http_header_index_t;
typedef struct lsapi_header_offset lsapi_header_offset_t;
typedef struct lsapi_packet_header lsapi_packet_header_t;
typedef struct lsapi_resp_info lsapi_resp_info;

// lscapi opaque structures
struct lscapi_rec;
struct lsphp_conn_t;

typedef struct lscapi_rec lscapi_rec;
typedef struct lsphp_conn_t lsphp_conn_t;

// allocator function pointer type
typedef void *(*lscapi_alloc_fn)(size_t size, void *user_data);

typedef long (*lscapi_user_get_body_fn)(char *buf, size_t bufLen, void *user_data);

typedef int (*lscapi_should_get_block_fn)(void *user_data);

typedef void (*lscapi_log_fn)(const char *file, int line, int level, int errcode,
                                const void *user_data, const char *fmt, ...);

typedef void (*lscapi_random_fn)(void *buf, size_t size);

typedef int(*lsphp_conn_failed_fn)(lsphp_conn_t *conn);

#ifndef WITHOUT_LVE
typedef void* (*lscapi_lve_alloc_t)(size_t sz, void* user_data);
typedef void (*lscapi_lve_register_t)(void* user_data, void* data, int (*destroy_fn)());
#endif

struct lscapi_var_t
{
    const char *key;
    int keyLen;
    const char *val;
    int valLen;
    int perdir;
};
typedef struct lscapi_var_t lscapi_var_t;


struct lscapi_header_index_t
{
    unsigned keyOff;
    unsigned keyLen;
    unsigned valOff;
    unsigned valLen;
};
typedef struct lscapi_header_index_t lscapi_header_index_t;


/*
  From lsapilib.h of Litespeed API 6.11
  In Litespeed API 7.4.2 increased to 1000

  DEPRECATED
  For backward compatibility only.
  It it needed only for deprecated lscapi_resphdr_t
*/
#define LSAPI_MAX_RESP_HEADERS  100


/*
  DEPRECATED
  For backward compatibility only.
  It it needed only for deprecated lscapi_receive_response_hdr function
*/
struct lscapi_resphdr_t
{
    char buf[LSAPI_MAX_RESP_HEADERS * (LSAPI_RESP_HTTP_HEADER_MAX+2) + 1024];
    size_t dataLen;
    uint16_t *hdrSizes;
    int hdrNum;
    int respStatus;
};
typedef struct lscapi_resphdr_t lscapi_resphdr_t;

struct lscapi_resp_hdr_t
{
    char *respBuffer;
    size_t dataLen;
    uint16_t *hdrSizes;
    int hdrNum;
    int respStatus;
};
typedef struct lscapi_resp_hdr_t lscapi_resp_hdr_t;

struct lscapi_backend_info_t
{
    uint32_t connect_tries;
    uint32_t connect_timeout;

    uint32_t backend_children;
    uint32_t backend_pgrp_max_idle;
    uint32_t backend_max_idle;
    uint32_t backend_max_process_time;
    uint32_t backend_max_reqs;
    uint32_t backend_pgrp_max_reqs;
    uint32_t backend_pgrp_max_crashes;


    uint32_t backend_initial_start;
    uint32_t poll_timeout;

    int32_t backend_oom_score_adj;

    unsigned backend_coredump: 1;
    unsigned dump_backend_debug_info: 1;
    unsigned use_suexec: 1;
    unsigned enable_user_ini: 1;
    unsigned per_user: 1;
    unsigned user_ini_homedir: 1;

    unsigned backend_coredump_was_set: 1;
    unsigned dump_backend_debug_info_was_set: 1;
    unsigned use_suexec_was_set: 1;
    unsigned enable_user_ini_was_set: 1;
    unsigned per_user_was_set: 1;
    unsigned user_ini_homedir_was_set: 1;

    unsigned avoid_zombies: 1;
    unsigned avoid_zombies_was_set: 1;

    unsigned disable_reject_mode: 1;
    unsigned disable_reject_mode_was_set: 1;

    unsigned use_own_log: 1;
    unsigned use_own_log_was_set: 1;

    unsigned common_own_log: 1;
    unsigned common_own_log_was_set: 1;

    unsigned pwd_disabled: 1;
    unsigned pwd_disabled_was_set: 1;

    unsigned backend_loglevel_info: 1;
    unsigned backend_loglevel_info_was_set: 1;

    unsigned backend_oom_score_adj_was_set: 1;

};
typedef struct lscapi_backend_info_t lscapi_backend_info_t;


#define LSCAPI_PATH_MAX 128
#define LSCAPI_PHPRC_MAX 128
/*
 struct sockaddr_un in <sys/un.h> contains integer constant
 instead of promised by unix(7) UNIX_PATH_MAX
 So we will use our own macro.
*/
#define LSCAPI_SOCKET_MAX 108

typedef struct spawn_info_t
{
    unsigned backend_children;
    int backend_oom_score_adj;
    unsigned backend_pgrp_max_idle;
    unsigned backend_max_idle;
    unsigned backend_max_process_time;
    unsigned backend_max_reqs;
    unsigned backend_pgrp_max_reqs;
    unsigned backend_pgrp_max_crashes;
    unsigned backend_initial_start;
    uid_t real_uid;
    gid_t real_gid;
    uid_t lve_uid;
    unsigned backend_coredump: 1;
    unsigned dump_backend_debug_info: 1;
    unsigned use_suexec: 1;
    unsigned backend_accept_notify: 1;
    unsigned use_own_log: 1;
    unsigned enable_user_ini: 1;
    unsigned common_own_log: 1;
    unsigned user_ini_homedir: 1;
    char phprc[LSCAPI_PHPRC_MAX];
    char cmd[LSCAPI_PATH_MAX];
    char socket_name[LSCAPI_SOCKET_MAX];
    unsigned pwd_disabled: 1;
    unsigned avoid_zombies: 1;
    unsigned disable_reject_mode: 1;
    char uniq_socket_name[LSCAPI_SOCKET_MAX];
} spawn_info_t;

typedef struct spawn_context_t
{
    int log_sock;
    int restore;
} spawn_context_t;


typedef pid_t (*lscapi_spawn_backend_fn)(const spawn_info_t *spawn_info,
                                         void *user_data, int force_start);

// Event masks

#define LSCAPI_RESPONSE_FINISHED       (1u << 1)

#define LSCAPI_SENDREQ_INTERNAL_ERROR  (1u << 3)
#define LSCAPI_SENDREQ_BACKEND_ERROR   (1u << 4)
#define LSCAPI_SENDREQ_CLIENT_ERROR    (1u << 5)
#define LSCAPI_CONNECT_REJECTED        (1u << 6)
#define LSCAPI_CONNECT_GENERIC_ERROR   (1u << 7)
#define LSCAPI_CONNECT_RESET_ERROR     (1u << 8)

enum lscapi_create_flags
{
	LSCAPI_CREATE_WITH_LVE	= 1 << 0,
	LSCAPI_OUT_ALREADY_INSIDE_LVE	= 1 << 16,
	LSCAPI_OUT_RESOURCE_LIMIT_LVE	= 1 << 17,
};

/*
 *  For backward compatibility with mod_lsapi
 *   LSCAPI_SPECIAL_ENV_PHPVAL_SYSTEM should be 0
 *    LSCAPI_SPECIAL_ENV_PHPVAL_PERDIR should be 1
 *    */
enum lscapi_special_env_types
{
    LSCAPI_SPECIAL_ENV_PHPVAL_SYSTEM = 0,
    LSCAPI_SPECIAL_ENV_PHPVAL_PERDIR = 1,
    LSCAPI_SPECIAL_ENV_ALTER_PHPINI = 2
};

extern const char defaultHandler[]; // "application/x-httpd-lsphp";

int lscapi_init(char *errbuf, size_t errlen);

// Here caller must use memory pool associated with process
void lscapi_child_init(lscapi_alloc_fn user_alloc, void *user_data);

// for backward compatibility, use lscapi_get_backend_heuristic instead
const char *lscapi_get_backend(const char *handler);

// Here caller must use memory pool associated with HTTP request
const char *lscapi_get_backend_heuristic(const char *handler, lscapi_alloc_fn user_alloc, void *user_data);

uint32_t lscapi_get_real_uid(lscapi_rec *lscapi);

/**
 * create opaque structure to store connection to lsphp on logical level
 * args:
 * user_alloc = pointer to allocator function
 * user_data = arbitrary user data to pass to allocator function
 * real_uid = uid for suexec
 * real_gid = gid for suexec
 * lve_uid = uid for lve
 * flags =
 * errbuf = buffer to store error description on error
 * errlen = length of the error buffer
 * return codes:
 * allocated and initialized structure = on success, NULL on error
 */
lscapi_rec *lscapi_create_ex(lscapi_alloc_fn user_alloc,
                       lscapi_log_fn user_log,
                       void *user_data,
                       uint32_t real_uid, uint32_t real_gid, uint32_t lve_uid,
                       unsigned *flagsPtr,
                       char *errbuf, unsigned errlen);


/**
 * close connection to lsphp on logical level
 * args:
 * lscapi = opaque structure created with lscapi_create
 */
void lscapi_destroy(lscapi_rec *lscapi);

/**
 * return codes:
 * 0 = on success, -1 on error and sets appropriate flag on eventMaskPtr
 */
int lscapi_send_request(lsphp_conn_t *conn, unsigned *eventMaskPtr,
                        char *errbuf, unsigned errlen);

void lscapi_set_socket_path(lscapi_rec *lscapi, char *socket_path);
void lscapi_set_phprc(lscapi_rec *lscapi, const char *phprc);

void lscapi_set_user_body_info(lscapi_rec *lscapi, size_t body_len,
                               lscapi_user_get_body_fn user_get_body,
                               lscapi_should_get_block_fn user_should_get_block);

void lscapi_set_use_request_scope_buffer(lscapi_rec *lscapi, bool value);

size_t lscapi_get_body_len(lscapi_rec *lscapi);

void lscapi_set_header_info(lscapi_rec *lscapi,
                            lsapi_http_header_index_t *hdrIndexPtr,
                            lsapi_header_offset_t *hdrOffsets,
                            size_t hdrOffsetsNum,
                            char *httpHeader,
                            size_t httpHeaderLen);

void lscapi_set_envs(lscapi_rec *lscapi, lscapi_var_t *envs, int envNum);
void lscapi_set_special_envs(lscapi_rec *lscapi, lscapi_var_t *specialEnvs, int specialEnvNum);

void lscapi_set_backend_path(lscapi_rec *lscapi,
                             const char *handler,
                             const char *backend_path);

void lscapi_set_debug_enabled(lscapi_rec *lscapi, int debug_enabled);

void lscapi_set_error(lscapi_rec *lscapi);
void lscapi_set_recoverable_error(lscapi_rec *lscapi);
bool lscapi_get_error(const lscapi_rec *lscapi, bool *outIsRecoverableError);
void lscapi_reset_internal_error_state(lscapi_rec *lscapi);

void lscapi_set_random(lscapi_rec *lscapi, lscapi_random_fn get_random_bytes);

void lscapi_set_user_spawn_backend(lscapi_rec *lscapi, lscapi_spawn_backend_fn user_spawn_backend);

void lscapi_set_accept_notify(lscapi_rec *lscapi, int accept_notify);

void lscapi_set_domain(lscapi_rec *lscapi,
                        const char *domain);

void lscapi_set_protocol_log(lscapi_rec *lscapi,
                                const char *protocol_log);

void lscapi_set_tmpdir(lscapi_rec *lscapi, const char *tmpdir);


void lscapi_init_backend_info(lscapi_backend_info_t *infoPtr);
void lscapi_merge_backend_info(lscapi_backend_info_t *base, lscapi_backend_info_t *cur, lscapi_backend_info_t *merged);

lsphp_conn_t* lscapi_acquire_lsphp_conn(lscapi_rec *lscapi, char *errbuf, unsigned errlen);

void lscapi_release_lsphp_conn(lsphp_conn_t *conn);


/*
   This function creates a safe environment for spawning new lsphp process.
   Memory for this is allocated with call to calloc and calls to strdup.
   So this function is intended to be called just after call to fork from within a child process and just before call to exec.
   DO NOT USE it otherwise to avoid memory leaks.
*/
char** lscapi_prepare_env(spawn_info_t *spawn_info, const char *customEnv, int envLen, int envNum, const char *safePath, char *errbuf, unsigned errlen);

int lscapi_lsphp_conn_get_socket(lsphp_conn_t *conn);

void lscapi_remove_stale_data_from_socket(lsphp_conn_t *conn);

int lscapi_is_conn_closed(lsphp_conn_t *conn);



int lscapi_determine_conn_lsphp(lsphp_conn_t *conn,
                                lscapi_backend_info_t *backendInfoPtr,
                                char *errbuf, unsigned errlen);

int lscapi_determine_conn_lsphp_ex(lsphp_conn_t *conn,
                                    lscapi_backend_info_t *backendInfoPtr,
                                    const char *sockName,
                                    char *errbuf, unsigned errlen);

int lscapi_connect_backend(lsphp_conn_t *conn, unsigned *eventMaskPtr, char *errbuf, unsigned errlen);
int lscapi_connect_backend_ex(lsphp_conn_t *conn, unsigned *eventMaskPtr, char *errbuf, unsigned errlen, int chunked);

void lscapi_conn_set_failed_cb(lsphp_conn_t *conn, lsphp_conn_failed_fn cb, void *opaque);
lscapi_rec *lscapi_conn_get_lsapi_handle(lsphp_conn_t *conn);
void *lscapi_conn_get_opaque(lsphp_conn_t *conn);

const char *lscapi_conn_get_socket_name(lsphp_conn_t *conn);
pid_t lscapi_conn_get_worker_pid(lsphp_conn_t *conn);

/* DEPRECATED */
void lscapi_lsphp_use_sock(lsphp_conn_t *conn, int sock, char *errbuf, unsigned errlen);

void lscapi_lsphp_use_sock_ext(lsphp_conn_t *conn, int sock, pid_t backend_pid, char *errbuf, unsigned errlen);


/*
  DEPRECATED
  For backward compatibility only.
  Use lscapi_receive_resp_hdr and lscapi_free_resp_hdr instead
*/
int lscapi_receive_response_hdr(lsphp_conn_t *conn,
                                lscapi_resphdr_t *hdrInfoPtr,
                                unsigned *eventMaskPtr,
                                char *errbuf, unsigned errlen);

/*
    Use it instead of deprecated lscapi_receive_response_hdr
    to receive header of http response from backend.

    lscapi_free_resp_hdr should called to free allocated resources.
*/
int lscapi_receive_resp_hdr(lsphp_conn_t *conn,
                            lscapi_resp_hdr_t *rspHdr,
                            unsigned *eventMaskPtr,
                            char *errbuf, unsigned errlen);

void lscapi_free_resp_hdr(lsphp_conn_t *conn,
                          lscapi_resp_hdr_t *rspHdr);


int lscapi_receive_response_chunk(lsphp_conn_t *conn,
                                  char *buf, size_t *lenPtr,
                                  unsigned *eventMaskPtr,
                                  char *errbuf, unsigned errlen);


/*
    socket_path - sockets with patterns lsapi_*.sock in this directory will be deleted after stop of backends
*/
void lscapi_terminate_backends_ex(const char *socket_path);



void lscapi_sign_uidgid_md5(lscapi_rec *lscapi, const uint32_t ids[2],
                            const uint8_t salt[LSCAPI_SALT_SIZE],
                            unsigned char sign[32]);

void lscapi_sign_uidgid_sha1(lscapi_rec *lscapi, const uint32_t ids[2],
                             const uint8_t salt[LSCAPI_SALT_SIZE],
                             uint8_t sign[LSCAPI_SHA1_SIGN_SIZE]);

void lscapi_sign_string_sha1(lscapi_rec *lscapi, const char *to_sign,
                             const uint8_t salt[LSCAPI_SALT_SIZE],
                             uint8_t sign[LSCAPI_SHA1_SIGN_SIZE]);

void lscapi_encode_bytes(const uint8_t *bytes, size_t len, char *buf);

void lscapi_encode_sha1_sign_without_salt(const uint8_t sign[LSCAPI_SHA1_SIGN_SIZE], char *buf);

pid_t lscapi_spawn_backend(spawn_info_t *spawn_info, const spawn_context_t *spawn_context);


#define LSCAPI_PROC_CREATE_EXPORTED
pid_t lscapi_proc_create(const char * const *args,
                         const char * const *env );


/*
#define PROFILE_ENABLED
#ifndef PROFILE_LOG
#define PROFILE_LOG
#endif
*/

#ifdef PROFILE_ENABLED

#include <sys/time.h>
#include <sys/resource.h>

extern __thread int PROFILE_counter;
extern char *PROFILE_margins[];
#ifdef PROFILE_LOG

#define PROFILE_START(pr) \
    static __thread struct timeval pr##_U##__LINE__, pr##_S##__LINE__; \
    static __thread int32_t pr##_CUMU##__LINE__, pr##_CUMS##__LINE__; \
    static __thread int pr##_CNT##__LINE__; \
    static __thread int pr##_ENTERED##__LINE__; \
do { \
    struct rusage ruse; \
    getrusage(RUSAGE_THREAD, &ruse); \
    pr##_U##__LINE__ = ruse.ru_utime; \
    pr##_S##__LINE__ = ruse.ru_stime; \
    (pr##_CNT##__LINE__)++; \
    if(!((pr##_CNT##__LINE__) % 100))  {  \
        fprintf(stderr, "%s%s.%d: " #pr ": ENTER\n", \
                PROFILE_margins[PROFILE_counter], __FUNCTION__, __LINE__ ); \
        PROFILE_counter++; \
        pr##_ENTERED##__LINE__ = 1; \
    } \
} while(0)

#define PROFILE_STOP(pr) do { \
    struct rusage ruse; \
    struct timeval res_utime, res_stime; \
    getrusage(RUSAGE_THREAD, &ruse); \
    timersub(&ruse.ru_utime, &(pr##_U##__LINE__), &res_utime); \
    timersub(&ruse.ru_stime, &(pr##_S##__LINE__), &res_stime); \
    pr##_CUMU##__LINE__ += res_utime.tv_sec * 1000000 + res_utime.tv_usec; \
    pr##_CUMS##__LINE__ += res_stime.tv_sec * 1000000 + res_stime.tv_usec; \
    if(pr##_ENTERED##__LINE__) { \
        pr##_ENTERED##__LINE__ = 0; \
        PROFILE_counter--; \
        double u = pr##_CUMU##__LINE__; u /= pr##_CNT##__LINE__; \
        double s = pr##_CUMS##__LINE__; s /= pr##_CNT##__LINE__; \
        fprintf(stderr, "%s%s.%d: " #pr ": u:%g; s:%g; cnt %d\n", \
                PROFILE_margins[PROFILE_counter], __FUNCTION__, __LINE__, u, s, pr##_CNT##__LINE__ ); \
    } \
} while(0)

#else // PROFILE_LOG

#define PROFILE_START(pr)
#define PROFILE_STOP(pr)

#endif // PROFILE_LOG

#else // PROFILE_ENABLED

#define PROFILE_START(pr)
#define PROFILE_STOP(pr)

#ifdef __clang__
#define ENVIRONMENT __environ
#else
#define ENVIRONMENT environ
#endif

#endif // PROFILE_ENABLED

#endif // _LSCAPI_H_
