<?php

// {{{ license

/* vim: set expandtab tabstop=4 shiftwidth=4 softtabstop=4 foldmethod=marker: */
//
// +----------------------------------------------------------------------+
// | This library is free software; you can redistribute it and/or modify |
// | it under the terms of the GNU Lesser General Public License as       |
// | published by the Free Software Foundation; either version 2.1 of the |
// | License, or (at your option) any later version.                      |
// |                                                                      |
// | This library is distributed in the hope that it will be useful, but  |
// | WITHOUT ANY WARRANTY; without even the implied warranty of           |
// | MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    |
// | Lesser General Public License for more details.                      |
// |                                                                      |
// | You should have received a copy of the GNU Lesser General Public     |
// | License along with this library; if not, write to the Free Software  |
// | Foundation, Inc., 51 Franklin St, Boston, MA 02110, United States    |
// +----------------------------------------------------------------------+
//
// }}}

/**
 * Encode/decode Internationalized Domain Names.
 *
 * The class allows to convert internationalized domain names
 * (see RFC 3490 for details) as they can be used with various registries worldwide
 * to be translated between their original (localized) form and their encoded form
 * as it will be used in the DNS (Domain Name System).
 *
 * The class provides two public methods, encode() and decode(), which do exactly
 * what you would expect them to do. You are allowed to use complete domain names,
 * simple strings and complete email addresses as well. That means, that you might
 * use any of the following notations:
 *
 * - www.nörgler.com
 * - xn--nrgler-wxa
 * - xn--brse-5qa.xn--knrz-1ra.info
 *
 * Unicode input might be given as either UTF-8 string, UCS-4 string or UCS-4 array.
 * Unicode output is available in the same formats.
 * You can select your preferred format via {@link set_paramter()}.
 *
 * ACE input and output is always expected to be ASCII.
 *
 * @author  Matthias Sommerfeld <mso@phlylabs.de>
 * @copyright 2004-2016 phlyLabs Berlin, http://phlylabs.de
 * @version 0.9.1 2016-01-06
 */
class idna_convert {

    private $version = '0.9.1';
    protected $sub_version = 'main';

    // NP See below
    // Internal settings, do not mess with them
    protected $_punycode_prefix = 'xn--';
    protected $_invalid_ucs = 0x80000000;
    protected $_max_ucs = 0x10FFFF;
    protected $_base = 36;
    protected $_tmin = 1;
    protected $_tmax = 26;
    protected $_skew = 38;
    protected $_damp = 700;
    protected $_initial_bias = 72;
    protected $_initial_n = 0x80;
    protected $_sbase = 0xAC00;
    protected $_lbase = 0x1100;
    protected $_vbase = 0x1161;
    protected $_tbase = 0x11A7;
    protected $_lcount = 19;
    protected $_vcount = 21;
    protected $_tcount = 28;
    protected $_ncount = 588;   // _vcount * _tcount
    protected $_scount = 11172; // _lcount * _tcount * _vcount
    protected $_error = false;
    protected static $_mb_string_overload = null;
    // See {@link set_paramter()} for details of how to change the following
    // settings from within your script / application
    protected $_api_encoding = 'utf8';   // Default input charset is UTF-8
    protected $_allow_overlong = false;  // Overlong UTF-8 encodings are forbidden
    protected $_strict_mode = false;     // Behave strict or not
    protected $_idn_version = 2003;      // Can be either 2003 (old, default) or 2008

    /**
     * the constructor
     *
     * @param array $options
     * @return boolean
     * @since 0.5.2
     */
    public function __construct($options = false)
    {
        // If parameters are given, pass these to the respective method
        if (is_array($options)) {
            $this->set_parameter($options);
        }

        // populate mbstring overloading cache if not set
        if (self::$_mb_string_overload === null) {
            self::$_mb_string_overload = (extension_loaded('mbstring') && (ini_get('mbstring.func_overload') & 0x02) === 0x02);
        }
    }

    public function get_version()
    {
        return $this->version.'-'.$this->sub_version;
    }

    /**
     * Sets a new option value. Available options and values:
     * [encoding - Use either UTF-8, UCS4 as array or UCS4 as string as input ('utf8' for UTF-8,
     *         'ucs4_string' and 'ucs4_array' respectively for UCS4); The output is always UTF-8]
     * [overlong - Unicode does not allow unnecessarily long encodings of chars,
     *             to allow this, set this parameter to true, else to false;
     *             default is false.]
     * [strict - true: strict mode, good for registration purposes - Causes errors
     *           on failures; false: loose mode, ideal for "wildlife" applications
     *           by silently ignoring errors and returning the original input instead
     *
     * @param    mixed     Parameter to set (string: single parameter; array of Parameter => Value pairs)
     * @param    string    Value to use (if parameter 1 is a string)
     * @return   boolean   true on success, false otherwise
     */
    public function set_parameter($option, $value = false)
    {
        if (!is_array($option)) {
            $option = array($option => $value);
        }
        foreach ($option as $k => $v) {
            switch ($k) {
                case 'encoding':
                    switch ($v) {
                        case 'utf8':
                        case 'ucs4_string':
                        case 'ucs4_array':
                            $this->_api_encoding = $v;
                            break;
                        default:
                            $this->_error('Set Parameter: Unknown parameter ' . $v . ' for option ' . $k);
                            return false;
                    }
                    break;
                case 'overlong':
                    $this->_allow_overlong = ($v) ? true : false;
                    break;
                case 'strict':
                    $this->_strict_mode = ($v) ? true : false;
                    break;
                case 'idn_version':
                    if (in_array($v, array('2003', '2008'))) {
                        $this->_idn_version = $v;
                    } else {
                        $this->_error('Set Parameter: Unknown parameter ' . $v . ' for option ' . $k);
                    }
                    break;
                case 'encode_german_sz': // Deprecated
                    if (!$v) {
                        self::$NP['replacemaps'][0xDF] = array(0x73, 0x73);
                    } else {
                        unset(self::$NP['replacemaps'][0xDF]);
                    }
                    break;
                default:
                    $this->_error('Set Parameter: Unknown option ' . $k);
                    return false;
            }
        }
        return true;
    }

    /**
     * Decode a given ACE domain name
     * @param    string   Domain name (ACE string)
     * [@param    string   Desired output encoding, see {@link set_parameter}]
     * @return   string   Decoded Domain name (UTF-8 or UCS-4)
     */
    public function decode($input, $one_time_encoding = false)
    {
        // Optionally set
        if ($one_time_encoding) {
            switch ($one_time_encoding) {
                case 'utf8':
                case 'ucs4_string':
                case 'ucs4_array':
                    break;
                default:
                    $this->_error('Unknown encoding ' . $one_time_encoding);
                    return false;
            }
        }
        // Make sure to drop any newline characters around
        $input = trim($input);

        // Negotiate input and try to determine, whether it is a plain string,
        // an email address or something like a complete URL
        if (strpos($input, '@')) { // Maybe it is an email address
            // No no in strict mode
            if ($this->_strict_mode) {
                $this->_error('Only simple domain name parts can be handled in strict mode');
                return false;
            }
            list ($email_pref, $input) = explode('@', $input, 2);
            $arr = explode('.', $input);
            foreach ($arr as $k => $v) {
                if (preg_match('!^' . preg_quote($this->_punycode_prefix, '!') . '!', $v)) {
                    $conv = $this->_decode($v);
                    if ($conv) {
                        $arr[$k] = $conv;
                    }
                }
            }
            $input = join('.', $arr);
            $arr = explode('.', $email_pref);
            foreach ($arr as $k => $v) {
                if (preg_match('!^' . preg_quote($this->_punycode_prefix, '!') . '!', $v)) {
                    $conv = $this->_decode($v);
                    if ($conv) {
                        $arr[$k] = $conv;
                    }
                }
            }
            $email_pref = join('.', $arr);
            $return = $email_pref . '@' . $input;
        } elseif (preg_match('![:\./]!', $input)) { // Or a complete domain name (with or without paths / parameters)
            // No no in strict mode
            if ($this->_strict_mode) {
                $this->_error('Only simple domain name parts can be handled in strict mode');
                return false;
            }
            $parsed = parse_url($input);
            if (isset($parsed['host'])) {
                $arr = explode('.', $parsed['host']);
                foreach ($arr as $k => $v) {
                    $conv = $this->_decode($v);
                    if ($conv) {
                        $arr[$k] = $conv;
                    }
                }
                $parsed['host'] = join('.', $arr);
                $return = (empty($parsed['scheme']) ? '' : $parsed['scheme'] . (strtolower($parsed['scheme']) == 'mailto' ? ':' : '://')).
                        (empty($parsed['user']) ? '' : $parsed['user'] . (empty($parsed['pass']) ? '' : ':' . $parsed['pass']) . '@').
                        $parsed['host'].
                        (empty($parsed['port']) ? '' : ':' . $parsed['port']).
                        (empty($parsed['path']) ? '' : $parsed['path']).
                        (empty($parsed['query']) ? '' : '?' . $parsed['query']).
                        (empty($parsed['fragment']) ? '' : '#' . $parsed['fragment']);
            } else { // parse_url seems to have failed, try without it
                $arr = explode('.', $input);
                foreach ($arr as $k => $v) {
                    $conv = $this->_decode($v);
                    $arr[$k] = ($conv) ? $conv : $v;
                }
                $return = join('.', $arr);
            }
        } else { // Otherwise we consider it being a pure domain name string
            $return = $this->_decode($input);
            if (!$return) {
                $return = $input;
            }
        }
        // The output is UTF-8 by default, other output formats need conversion here
        // If one time encoding is given, use this, else the objects property
        switch (($one_time_encoding) ? $one_time_encoding : $this->_api_encoding) {
            case 'utf8':        return $return; // break;
            case 'ucs4_string': return $this->_ucs4_to_ucs4_string($this->_utf8_to_ucs4($return));  // break;
            case 'ucs4_array':  return $this->_utf8_to_ucs4($return); // break;
            default:            $this->_error('Unsupported output format'); return false;
        }
    }

    /**
     * Encode a given UTF-8 domain name
     * @param    string   Domain name (UTF-8 or UCS-4)
     * [@param    string   Desired input encoding, see {@link set_parameter}]
     * @return   string   Encoded Domain name (ACE string)
     */
    public function encode($decoded, $one_time_encoding = false)
    {
        // Forcing conversion of input to UCS4 array
        // If one time encoding is given, use this, else the objects property
        switch ($one_time_encoding ? $one_time_encoding : $this->_api_encoding) {
            case 'utf8':
                $decoded = $this->_utf8_to_ucs4($decoded);
                break;
            case 'ucs4_string':
                $decoded = $this->_ucs4_string_to_ucs4($decoded);
            case 'ucs4_array':
                break;
            default:
                $this->_error('Unsupported input format: ' . ($one_time_encoding ? $one_time_encoding : $this->_api_encoding));
                return false;
        }

        // No input, no output, what else did you expect?
        if (empty($decoded)) {
            return '';
        }

        // Anchors for iteration
        $last_begin = 0;
        // Output string
        $output = '';
        foreach ($decoded as $k => $v) {
            // Make sure to use just the plain dot
            switch ($v) {
                case 0x3002:
                case 0xFF0E:
                case 0xFF61:
                    $decoded[$k] = 0x2E;
                    // Right, no break here, the above are converted to dots anyway
                // Stumbling across an anchoring character
                case 0x2E:
                case 0x2F:
                case 0x3A:
                case 0x3F:
                case 0x40:
                    // Neither email addresses nor URLs allowed in strict mode
                    if ($this->_strict_mode) {
                        $this->_error('Neither email addresses nor URLs are allowed in strict mode.');
                        return false;
                    } else {
                        // Skip first char
                        if ($k) {
                            $encoded = '';
                            $encoded = $this->_encode(array_slice($decoded, $last_begin, (($k) - $last_begin)));
                            if ($encoded) {
                                $output .= $encoded;
                            } else {
                                $output .= $this->_ucs4_to_utf8(array_slice($decoded, $last_begin, (($k) - $last_begin)));
                            }
                            $output .= chr($decoded[$k]);
                        }
                        $last_begin = $k + 1;
                    }
            }
        }
        // Catch the rest of the string
        if ($last_begin) {
            $inp_len = sizeof($decoded);
            $encoded = '';
            $encoded = $this->_encode(array_slice($decoded, $last_begin, (($inp_len) - $last_begin)));
            if ($encoded) {
                $output .= $encoded;
            } else {
                $output .= $this->_ucs4_to_utf8(array_slice($decoded, $last_begin, (($inp_len) - $last_begin)));
            }
            return $output;
        } else {
            if (false !== ($output = $this->_encode($decoded))) {
                return $output;
            } else {
                return $this->_ucs4_to_utf8($decoded);
            }
        }
    }

    /**
     * Removes a weakness of encode(), which cannot properly handle URIs but instead encodes their
     * path or query components, too.
     * @param string  $uri  Expects the URI as a UTF-8 (or ASCII) string
     * @return  string  The URI encoded to Punycode, everything but the host component is left alone
     * @since 0.6.4
     */
    public function encode_uri($uri)
    {
        $parsed = parse_url($uri);
        if (!isset($parsed['host'])) {
            $this->_error('The given string does not look like a URI');
            return false;
        }
        $arr = explode('.', $parsed['host']);
        foreach ($arr as $k => $v) {
            $conv = $this->encode($v, 'utf8');
            if ($conv) {
                $arr[$k] = $conv;
            }
        }
        $parsed['host'] = join('.', $arr);
        $return = (empty($parsed['scheme']) ? '' : $parsed['scheme'] . (strtolower($parsed['scheme']) == 'mailto' ? ':' : '://')).
                (empty($parsed['user']) ? '' : $parsed['user'] . (empty($parsed['pass']) ? '' : ':' . $parsed['pass']) . '@').
                $parsed['host'].
                (empty($parsed['port']) ? '' : ':' . $parsed['port']).
                (empty($parsed['path']) ? '' : $parsed['path']).
                (empty($parsed['query']) ? '' : '?' . $parsed['query']).
                (empty($parsed['fragment']) ? '' : '#' . $parsed['fragment']);
        return $return;
    }

    /**
     * Use this method to get the last error ocurred
     * @param    void
     * @return   string   The last error, that occured
     */
    public function get_last_error()
    {
        return $this->_error;
    }

    /**
     * The actual decoding algorithm
     * @param string
     * @return mixed
     */
    protected function _decode($encoded)
    {
        $decoded = array();
        // find the Punycode prefix
        if (!preg_match('!^' . preg_quote($this->_punycode_prefix, '!') . '!', $encoded)) {
            $this->_error('This is not a punycode string');
            return false;
        }
        $encode_test = preg_replace('!^' . preg_quote($this->_punycode_prefix, '!') . '!', '', $encoded);
        // If nothing left after removing the prefix, it is hopeless
        if (!$encode_test) {
            $this->_error('The given encoded string was empty');
            return false;
        }
        // Find last occurence of the delimiter
        $delim_pos = strrpos($encoded, '-');
        if ($delim_pos > self::byteLength($this->_punycode_prefix)) {
            for ($k = self::byteLength($this->_punycode_prefix); $k < $delim_pos; ++$k) {
                $decoded[] = ord($encoded[$k]);
            }
        }
        $deco_len = count($decoded);
        $enco_len = self::byteLength($encoded);

        // Wandering through the strings; init
        $is_first = true;
        $bias = $this->_initial_bias;
        $idx = 0;
        $char = $this->_initial_n;

        for ($enco_idx = ($delim_pos) ? ($delim_pos + 1) : 0; $enco_idx < $enco_len; ++$deco_len) {
            for ($old_idx = $idx, $w = 1, $k = $this->_base; 1; $k += $this->_base) {
                $digit = $this->_decode_digit($encoded[$enco_idx++]);
                $idx += $digit * $w;
                $t = ($k <= $bias) ? $this->_tmin :
                        (($k >= $bias + $this->_tmax) ? $this->_tmax : ($k - $bias));
                if ($digit < $t) {
                    break;
                }
                $w = (int) ($w * ($this->_base - $t));
            }
            $bias = $this->_adapt($idx - $old_idx, $deco_len + 1, $is_first);
            $is_first = false;
            $char += (int) ($idx / ($deco_len + 1));
            $idx %= ($deco_len + 1);
            if ($deco_len > 0) {
                // Make room for the decoded char
                for ($i = $deco_len; $i > $idx; $i--) {
                    $decoded[$i] = $decoded[($i - 1)];
                }
            }
            $decoded[$idx++] = $char;
        }
        return $this->_ucs4_to_utf8($decoded);
    }

    /**
     * The actual encoding algorithm
     * @param  string
     * @return mixed
     */
    protected function _encode($decoded)
    {
        // We cannot encode a domain name containing the Punycode prefix
        $extract = self::byteLength($this->_punycode_prefix);
        $check_pref = $this->_utf8_to_ucs4($this->_punycode_prefix);
        $check_deco = array_slice($decoded, 0, $extract);

        if ($check_pref == $check_deco) {
            $this->_error('This is already a punycode string');
            return false;
        }
        // We will not try to encode strings consisting of basic code points only
        $encodable = false;
        foreach ($decoded as $k => $v) {
            if ($v > 0x7a) {
                $encodable = true;
                break;
            }
        }
        if (!$encodable) {
            $this->_error('The given string does not contain encodable chars');
            return false;
        }
        // Do NAMEPREP
        $decoded = $this->_nameprep($decoded);
        if (!$decoded || !is_array($decoded)) {
            return false; // NAMEPREP failed
        }
        $deco_len = count($decoded);
        if (!$deco_len) {
            return false; // Empty array
        }
        $codecount = 0; // How many chars have been consumed
        $encoded = '';
        // Copy all basic code points to output
        for ($i = 0; $i < $deco_len; ++$i) {
            $test = $decoded[$i];
            // Will match [-0-9a-zA-Z]
            if ((0x2F < $test && $test < 0x40) || (0x40 < $test && $test < 0x5B) || (0x60 < $test && $test <= 0x7B) || (0x2D == $test)) {
                $encoded .= chr($decoded[$i]);
                $codecount++;
            }
        }
        if ($codecount == $deco_len) {
            return $encoded; // All codepoints were basic ones
        }
        // Start with the prefix; copy it to output
        $encoded = $this->_punycode_prefix . $encoded;
        // If we have basic code points in output, add an hyphen to the end
        if ($codecount) {
            $encoded .= '-';
        }
        // Now find and encode all non-basic code points
        $is_first = true;
        $cur_code = $this->_initial_n;
        $bias = $this->_initial_bias;
        $delta = 0;
        while ($codecount < $deco_len) {
            // Find the smallest code point >= the current code point and
            // remember the last ouccrence of it in the input
            for ($i = 0, $next_code = $this->_max_ucs; $i < $deco_len; $i++) {
                if ($decoded[$i] >= $cur_code && $decoded[$i] <= $next_code) {
                    $next_code = $decoded[$i];
                }
            }
            $delta += ($next_code - $cur_code) * ($codecount + 1);
            $cur_code = $next_code;

            // Scan input again and encode all characters whose code point is $cur_code
            for ($i = 0; $i < $deco_len; $i++) {
                if ($decoded[$i] < $cur_code) {
                    $delta++;
                } elseif ($decoded[$i] == $cur_code) {
                    for ($q = $delta, $k = $this->_base; 1; $k += $this->_base) {
                        $t = ($k <= $bias) ? $this->_tmin :
                                (($k >= $bias + $this->_tmax) ? $this->_tmax : $k - $bias);
                        if ($q < $t) {
                            break;
                        }
                        $encoded .= $this->_encode_digit(intval($t + (($q - $t) % ($this->_base - $t)))); //v0.4.5 Changed from ceil() to intval()
                        $q = (int) (($q - $t) / ($this->_base - $t));
                    }
                    $encoded .= $this->_encode_digit($q);
                    $bias = $this->_adapt($delta, $codecount + 1, $is_first);
                    $codecount++;
                    $delta = 0;
                    $is_first = false;
                }
            }
            $delta++;
            $cur_code++;
        }
        return $encoded;
    }

    /**
     * Adapt the bias according to the current code point and position
     * @param int $delta
     * @param int $npoints
     * @param int $is_first
     * @return int
     */
    protected function _adapt($delta, $npoints, $is_first)
    {
        $delta = intval($is_first ? ($delta / $this->_damp) : ($delta / 2));
        $delta += intval($delta / $npoints);
        for ($k = 0; $delta > (($this->_base - $this->_tmin) * $this->_tmax) / 2; $k += $this->_base) {
            $delta = intval($delta / ($this->_base - $this->_tmin));
        }
        return intval($k + ($this->_base - $this->_tmin + 1) * $delta / ($delta + $this->_skew));
    }

    /**
     * Encoding a certain digit
     * @param    int $d
     * @return string
     */
    protected function _encode_digit($d)
    {
        return chr($d + 22 + 75 * ($d < 26));
    }

    /**
     * Decode a certain digit
     * @param    int $cp
     * @return int
     */
    protected function _decode_digit($cp)
    {
        $cp = ord($cp);
        return ($cp - 48 < 10) ? $cp - 22 : (($cp - 65 < 26) ? $cp - 65 : (($cp - 97 < 26) ? $cp - 97 : $this->_base));
    }

    /**
     * Internal error handling method
     * @param  string $error
     */
    protected function _error($error = '')
    {
        $this->_error = $error;
    }

    /**
     * Do Nameprep according to RFC3491 and RFC3454
     * @param    array    Unicode Characters
     * @return   string   Unicode Characters, Nameprep'd
     */
    protected function _nameprep($input)
    {
        $output = array();
        //
        // Mapping
        // Walking through the input array, performing the required steps on each of
        // the input chars and putting the result into the output array
        // While mapping required chars we apply the cannonical ordering
        foreach ($input as $v) {
            // Map to nothing == skip that code point
            if (in_array($v, self::$NP['map_nothing'])) {
                continue;
            }
            // Try to find prohibited input
            if (in_array($v, self::$NP['prohibit']) || in_array($v, self::$NP['general_prohibited'])) {
                $this->_error('NAMEPREP: Prohibited input U+' . sprintf('%08X', $v));
                return false;
            }
            foreach (self::$NP['prohibit_ranges'] as $range) {
                if ($range[0] <= $v && $v <= $range[1]) {
                    $this->_error('NAMEPREP: Prohibited input U+' . sprintf('%08X', $v));
                    return false;
                }
            }

            if (0xAC00 <= $v && $v <= 0xD7AF) {
                // Hangul syllable decomposition
                foreach ($this->_hangul_decompose($v) as $out) {
                    $output[] = (int) $out;
                }
            } elseif (($this->_idn_version == '2003') && isset(self::$NP['replacemaps_2003'][$v])) {
                foreach ($this->_apply_cannonical_ordering(self::$NP['replacemaps_2003'][$v]) as $out) {
                    $output[] = (int) $out;
                }
            } elseif (($this->_idn_version == '2008') && isset(self::$NP['replacemaps'][$v])) {
                foreach ($this->_apply_cannonical_ordering(self::$NP['replacemaps'][$v]) as $out) {
                    $output[] = (int) $out;
                }
            } else {
                $output[] = (int) $v;
            }
        }
        // Before applying any Combining, try to rearrange any Hangul syllables
        $output = $this->_hangul_compose($output);
        //
        // Combine code points
        //
        $last_class = 0;
        $last_starter = 0;
        $out_len = count($output);
        for ($i = 0; $i < $out_len; ++$i) {
            $class = $this->_get_combining_class($output[$i]);
            if ((!$last_class || $last_class > $class) && $class) {
                // Try to match
                $seq_len = $i - $last_starter;
                $out = $this->_combine(array_slice($output, $last_starter, $seq_len));
                // On match: Replace the last starter with the composed character and remove
                // the now redundant non-starter(s)
                if ($out) {
                    $output[$last_starter] = $out;
                    if (count($out) != $seq_len) {
                        for ($j = $i + 1; $j < $out_len; ++$j) {
                            $output[$j - 1] = $output[$j];
                        }
                        unset($output[$out_len]);
                    }
                    // Rewind the for loop by one, since there can be more possible compositions
                    $i--;
                    $out_len--;
                    $last_class = ($i == $last_starter) ? 0 : $this->_get_combining_class($output[$i - 1]);
                    continue;
                }
            }
            // The current class is 0
            if (!$class) {
                $last_starter = $i;
            }
            $last_class = $class;
        }
        return $output;
    }

    /**
     * Decomposes a Hangul syllable
     * (see http://www.unicode.org/unicode/reports/tr15/#Hangul
     * @param    integer  32bit UCS4 code point
     * @return   array    Either Hangul Syllable decomposed or original 32bit value as one value array
     */
    protected function _hangul_decompose($char)
    {
        $sindex = (int) $char - $this->_sbase;
        if ($sindex < 0 || $sindex >= $this->_scount) {
            return array($char);
        }
        $result = array();
        $result[] = (int) $this->_lbase + $sindex / $this->_ncount;
        $result[] = (int) $this->_vbase + ($sindex % $this->_ncount) / $this->_tcount;
        $T = intval($this->_tbase + $sindex % $this->_tcount);
        if ($T != $this->_tbase) {
            $result[] = $T;
        }
        return $result;
    }

    /**
     * Ccomposes a Hangul syllable
     * (see http://www.unicode.org/unicode/reports/tr15/#Hangul
     * @param    array    Decomposed UCS4 sequence
     * @return   array    UCS4 sequence with syllables composed
     */
    protected function _hangul_compose($input)
    {
        $inp_len = count($input);
        if (!$inp_len) {
            return array();
        }
        $result = array();
        $last = (int) $input[0];
        $result[] = $last; // copy first char from input to output

        for ($i = 1; $i < $inp_len; ++$i) {
            $char = (int) $input[$i];
            $sindex = $last - $this->_sbase;
            $lindex = $last - $this->_lbase;
            $vindex = $char - $this->_vbase;
            $tindex = $char - $this->_tbase;
            // Find out, whether two current characters are LV and T
            if (0 <= $sindex && $sindex < $this->_scount && ($sindex % $this->_tcount == 0) && 0 <= $tindex && $tindex <= $this->_tcount) {
                // create syllable of form LVT
                $last += $tindex;
                $result[(count($result) - 1)] = $last; // reset last
                continue; // discard char
            }
            // Find out, whether two current characters form L and V
            if (0 <= $lindex && $lindex < $this->_lcount && 0 <= $vindex && $vindex < $this->_vcount) {
                // create syllable of form LV
                $last = (int) $this->_sbase + ($lindex * $this->_vcount + $vindex) * $this->_tcount;
                $result[(count($result) - 1)] = $last; // reset last
                continue; // discard char
            }
            // if neither case was true, just add the character
            $last = $char;
            $result[] = $char;
        }
        return $result;
    }

    /**
     * Returns the combining class of a certain wide char
     * @param    integer    Wide char to check (32bit integer)
     * @return   integer    Combining class if found, else 0
     */
    protected function _get_combining_class($char)
    {
        return isset(self::$NP['norm_combcls'][$char]) ? self::$NP['norm_combcls'][$char] : 0;
    }

    /**
     * Applies the cannonical ordering of a decomposed UCS4 sequence
     * @param    array      Decomposed UCS4 sequence
     * @return   array      Ordered USC4 sequence
     */
    protected function _apply_cannonical_ordering($input)
    {
        $swap = true;
        $size = count($input);
        while ($swap) {
            $swap = false;
            $last = $this->_get_combining_class(intval($input[0]));
            for ($i = 0; $i < $size - 1; ++$i) {
                $next = $this->_get_combining_class(intval($input[$i + 1]));
                if ($next != 0 && $last > $next) {
                    // Move item leftward until it fits
                    for ($j = $i + 1; $j > 0; --$j) {
                        if ($this->_get_combining_class(intval($input[$j - 1])) <= $next) {
                            break;
                        }
                        $t = intval($input[$j]);
                        $input[$j] = intval($input[$j - 1]);
                        $input[$j - 1] = $t;
                        $swap = true;
                    }
                    // Reentering the loop looking at the old character again
                    $next = $last;
                }
                $last = $next;
            }
        }
        return $input;
    }

    /**
     * Do composition of a sequence of starter and non-starter
     * @param    array      UCS4 Decomposed sequence
     * @return   array      Ordered USC4 sequence
     */
    protected function _combine($input)
    {
        $inp_len = count($input);
        if (0 == $inp_len) {
            return false;
        }
        foreach (self::$NP['replacemaps'] as $np_src => $np_target) {
            if ($np_target[0] != $input[0]) {
                continue;
            }
            if (count($np_target) != $inp_len) {
                continue;
            }
            $hit = false;
            foreach ($input as $k2 => $v2) {
                if ($v2 == $np_target[$k2]) {
                    $hit = true;
                } else {
                    $hit = false;
                    break;
                }
            }
            if ($hit) {
                return $np_src;
            }
        }
        return false;
    }

    /**
     * This converts an UTF-8 encoded string to its UCS-4 representation
     * By talking about UCS-4 "strings" we mean arrays of 32bit integers representing
     * each of the "chars". This is due to PHP not being able to handle strings with
     * bit depth different from 8. This apllies to the reverse method _ucs4_to_utf8(), too.
     * The following UTF-8 encodings are supported:
     * bytes bits  representation
     * 1        7  0xxxxxxx
     * 2       11  110xxxxx 10xxxxxx
     * 3       16  1110xxxx 10xxxxxx 10xxxxxx
     * 4       21  11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
     * 5       26  111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
     * 6       31  1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
     * Each x represents a bit that can be used to store character data.
     * The five and six byte sequences are part of Annex D of ISO/IEC 10646-1:2000
     * @param string $input
     * @return string
     */
    protected function _utf8_to_ucs4($input)
    {
        $output = array();
        $out_len = 0;
        $inp_len = self::byteLength($input);
        $mode = 'next';
        $test = 'none';
        for ($k = 0; $k < $inp_len; ++$k) {
            $v = ord($input[$k]); // Extract byte from input string
            if ($v < 128) { // We found an ASCII char - put into stirng as is
                $output[$out_len] = $v;
                ++$out_len;
                if ('add' == $mode) {
                    $this->_error('Conversion from UTF-8 to UCS-4 failed: malformed input at byte ' . $k);
                    return false;
                }
                continue;
            }
            if ('next' == $mode) { // Try to find the next start byte; determine the width of the Unicode char
                $start_byte = $v;
                $mode = 'add';
                $test = 'range';
                if ($v >> 5 == 6) { // &110xxxxx 10xxxxx
                    $next_byte = 0; // Tells, how many times subsequent bitmasks must rotate 6bits to the left
                    $v = ($v - 192) << 6;
                } elseif ($v >> 4 == 14) { // &1110xxxx 10xxxxxx 10xxxxxx
                    $next_byte = 1;
                    $v = ($v - 224) << 12;
                } elseif ($v >> 3 == 30) { // &11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
                    $next_byte = 2;
                    $v = ($v - 240) << 18;
                } elseif ($v >> 2 == 62) { // &111110xx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
                    $next_byte = 3;
                    $v = ($v - 248) << 24;
                } elseif ($v >> 1 == 126) { // &1111110x 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx 10xxxxxx
                    $next_byte = 4;
                    $v = ($v - 252) << 30;
                } else {
                    $this->_error('This might be UTF-8, but I don\'t understand it at byte ' . $k);
                    return false;
                }
                if ('add' == $mode) {
                    $output[$out_len] = (int) $v;
                    ++$out_len;
                    continue;
                }
            }
            if ('add' == $mode) {
                if (!$this->_allow_overlong && $test == 'range') {
                    $test = 'none';
                    if (($v < 0xA0 && $start_byte == 0xE0) || ($v < 0x90 && $start_byte == 0xF0) || ($v > 0x8F && $start_byte == 0xF4)) {
                        $this->_error('Bogus UTF-8 character detected (out of legal range) at byte ' . $k);
                        return false;
                    }
                }
                if ($v >> 6 == 2) { // Bit mask must be 10xxxxxx
                    $v = ($v - 128) << ($next_byte * 6);
                    $output[($out_len - 1)] += $v;
                    --$next_byte;
                } else {
                    $this->_error('Conversion from UTF-8 to UCS-4 failed: malformed input at byte ' . $k);
                    return false;
                }
                if ($next_byte < 0) {
                    $mode = 'next';
                }
            }
        } // for
        return $output;
    }

    /**
     * Convert UCS-4 string into UTF-8 string
     * See _utf8_to_ucs4() for details
     * @param string  $input
     * @return string
     */
    protected function _ucs4_to_utf8($input)
    {
        $output = '';
        foreach ($input as $k => $v) {
            if ($v < 128) { // 7bit are transferred literally
                $output .= chr($v);
            } elseif ($v < (1 << 11)) { // 2 bytes
                $output .= chr(192 + ($v >> 6)) . chr(128 + ($v & 63));
            } elseif ($v < (1 << 16)) { // 3 bytes
                $output .= chr(224 + ($v >> 12)) . chr(128 + (($v >> 6) & 63)) . chr(128 + ($v & 63));
            } elseif ($v < (1 << 21)) { // 4 bytes
                $output .= chr(240 + ($v >> 18)) . chr(128 + (($v >> 12) & 63)) . chr(128 + (($v >> 6) & 63)) . chr(128 + ($v & 63));
            } else {
                $this->_error('Conversion from UCS-4 to UTF-8 failed: malformed input at byte ' . $k);
                return false;
            }
        }
        return $output;
    }

    /**
     * Convert UCS-4 array into UCS-4 string
     *
     * @param array $input
     * @return string
     */
    protected function _ucs4_to_ucs4_string($input)
    {
        $output = '';
        // Take array values and split output to 4 bytes per value
        // The bit mask is 255, which reads &11111111
        foreach ($input as $v) {
            $output .= chr(($v >> 24) & 255) . chr(($v >> 16) & 255) . chr(($v >> 8) & 255) . chr($v & 255);
        }
        return $output;
    }

    /**
     * Convert UCS-4 strin into UCS-4 garray
     *
     * @param  string $input
     * @return array
     */
    protected function _ucs4_string_to_ucs4($input)
    {
        $output = array();
        $inp_len = self::byteLength($input);
        // Input length must be dividable by 4
        if ($inp_len % 4) {
            $this->_error('Input UCS4 string is broken');
            return false;
        }
        // Empty input - return empty output
        if (!$inp_len) {
            return $output;
        }
        for ($i = 0, $out_len = -1; $i < $inp_len; ++$i) {
            // Increment output position every 4 input bytes
            if (!($i % 4)) {
                $out_len++;
                $output[$out_len] = 0;
            }
            $output[$out_len] += ord($input[$i]) << (8 * (3 - ($i % 4) ) );
        }
        return $output;
    }

    /**
     * Gets the length of a string in bytes even if mbstring function
     * overloading is turned on
     *
     * @param string $string the string for which to get the length.
     * @return integer the length of the string in bytes.
     */
    protected static function byteLength($string)
    {
        if (self::$_mb_string_overload) {
            return mb_strlen($string, '8bit');
        }
        return strlen((binary) $string);
    }

    /**
     * Attempts to return a concrete IDNA instance.
     *
     * @param array $params Set of paramaters
     * @return idna_convert
     * @access public
     */
    public function getInstance($params = array())
    {
        return new idna_convert($params);
    }

    /**
     * Attempts to return a concrete IDNA instance for either php4 or php5,
     * only creating a new instance if no IDNA instance with the same
     * parameters currently exists.
     *
     * @param array $params Set of paramaters
     *
     * @return object idna_convert
     * @access public
     */
    public function singleton($params = array())
    {
        static $instances = array();

        $signature = serialize($params);
        if (!isset($instances[$signature])) {
            $instances[$signature] = idna_convert::getInstance($params);
        }
        return $instances[$signature];
    }

    /**
     * Holds all relevant mapping tables
     * See RFC3454 for details
     *
     * @private array
     * @since 0.5.2
     */
    protected static $NP = array(
            'map_nothing' => array(0xAD, 0x34F, 0x1806, 0x180B, 0x180C, 0x180D, 0x200B, 0x200C,
                    0x200D, 0x2060, 0xFE00, 0xFE01, 0xFE02, 0xFE03, 0xFE04, 0xFE05, 0xFE06, 0xFE07,
                    0xFE08, 0xFE09, 0xFE0A, 0xFE0B, 0xFE0C, 0xFE0D, 0xFE0E, 0xFE0F, 0xFEFF
            ),
            'general_prohibited' => array(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19,
                    20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42,
                    43, 44, 47, 59, 60, 61, 62, 63, 64, 91, 92, 93, 94, 95, 96, 123, 124, 125, 126, 127, 0x3002
            ),
            'prohibit' => array(0xA0, 0x340, 0x341, 0x6DD, 0x70F, 0x1680, 0x180E, 0x2000, 0x2001, 0x2002, 0x2003,
                    0x2004, 0x2005, 0x2006, 0x2007, 0x2008, 0x2009, 0x200A, 0x200B, 0x200C, 0x200D, 0x200E, 0x200F,
                    0x2028, 0x2029, 0x202A, 0x202B, 0x202C, 0x202D, 0x202E, 0x202F, 0x205F, 0x206A, 0x206B, 0x206C,
                    0x206D, 0x206E, 0x206F, 0x3000, 0x33C2, 0xFEFF, 0xFFF9, 0xFFFA, 0xFFFB, 0xFFFC, 0xFFFD, 0xFFFE, 0xFFFF,
                    0x1FFFE, 0x1FFFF, 0x2FFFE, 0x2FFFF, 0x3FFFE, 0x3FFFF, 0x4FFFE, 0x4FFFF, 0x5FFFE, 0x5FFFF, 0x6FFFE,
                    0x6FFFF, 0x7FFFE, 0x7FFFF, 0x8FFFE, 0x8FFFF, 0x9FFFE, 0x9FFFF, 0xAFFFE, 0xAFFFF, 0xBFFFE, 0xBFFFF,
                    0xCFFFE, 0xCFFFF, 0xDFFFE, 0xDFFFF, 0xE0001, 0xEFFFE, 0xEFFFF, 0xFFFFE, 0xFFFFF, 0x10FFFE, 0x10FFFF
            ),
            'prohibit_ranges' => array(array(0x80, 0x9F), array(0x2060, 0x206F), array(0x1D173, 0x1D17A),
                    array(0xE000, 0xF8FF), array(0xF0000, 0xFFFFD), array(0x100000, 0x10FFFD),
                    array(0xFDD0, 0xFDEF), array(0xD800, 0xDFFF), array(0x2FF0, 0x2FFB), array(0xE0020, 0xE007F)
            ),
            'replacemaps_2003' => array(0x41 => array(0x61), 0x42 => array(0x62), 0x43 => array(0x63),
                    0x44 => array(0x64), 0x45 => array(0x65), 0x46 => array(0x66), 0x47 => array(0x67),
                    0x48 => array(0x68), 0x49 => array(0x69), 0x4A => array(0x6A), 0x4B => array(0x6B),
                    0x4C => array(0x6C), 0x4D => array(0x6D), 0x4E => array(0x6E), 0x4F => array(0x6F),
                    0x50 => array(0x70), 0x51 => array(0x71), 0x52 => array(0x72), 0x53 => array(0x73),
                    0x54 => array(0x74), 0x55 => array(0x75), 0x56 => array(0x76), 0x57 => array(0x77),
                    0x58 => array(0x78), 0x59 => array(0x79), 0x5A => array(0x7A), 0xB5 => array(0x3BC),
                    0xC0 => array(0xE0), 0xC1 => array(0xE1), 0xC2 => array(0xE2), 0xC3 => array(0xE3),
                    0xC4 => array(0xE4), 0xC5 => array(0xE5), 0xC6 => array(0xE6), 0xC7 => array(0xE7),
                    0xC8 => array(0xE8), 0xC9 => array(0xE9), 0xCA => array(0xEA), 0xCB => array(0xEB),
                    0xCC => array(0xEC), 0xCD => array(0xED), 0xCE => array(0xEE), 0xCF => array(0xEF),
                    0xD0 => array(0xF0), 0xD1 => array(0xF1), 0xD2 => array(0xF2), 0xD3 => array(0xF3),
                    0xD4 => array(0xF4), 0xD5 => array(0xF5), 0xD6 => array(0xF6), 0xD8 => array(0xF8),
                    0xD9 => array(0xF9), 0xDA => array(0xFA), 0xDB => array(0xFB), 0xDC => array(0xFC),
                    0xDD => array(0xFD), 0xDE => array(0xFE), 0xDF => array(0x73, 0x73),
                    0x100 => array(0x101), 0x102 => array(0x103), 0x104 => array(0x105),
                    0x106 => array(0x107), 0x108 => array(0x109), 0x10A => array(0x10B),
                    0x10C => array(0x10D), 0x10E => array(0x10F), 0x110 => array(0x111),
                    0x112 => array(0x113), 0x114 => array(0x115), 0x116 => array(0x117),
                    0x118 => array(0x119), 0x11A => array(0x11B), 0x11C => array(0x11D),
                    0x11E => array(0x11F), 0x120 => array(0x121), 0x122 => array(0x123),
                    0x124 => array(0x125), 0x126 => array(0x127), 0x128 => array(0x129),
                    0x12A => array(0x12B), 0x12C => array(0x12D), 0x12E => array(0x12F),
                    0x130 => array(0x69, 0x307), 0x132 => array(0x133), 0x134 => array(0x135),
                    0x136 => array(0x137), 0x139 => array(0x13A), 0x13B => array(0x13C),
                    0x13D => array(0x13E), 0x13F => array(0x140), 0x141 => array(0x142),
                    0x143 => array(0x144), 0x145 => array(0x146), 0x147 => array(0x148),
                    0x149 => array(0x2BC, 0x6E), 0x14A => array(0x14B), 0x14C => array(0x14D),
                    0x14E => array(0x14F), 0x150 => array(0x151), 0x152 => array(0x153),
                    0x154 => array(0x155), 0x156 => array(0x157), 0x158 => array(0x159),
                    0x15A => array(0x15B), 0x15C => array(0x15D), 0x15E => array(0x15F),
                    0x160 => array(0x161), 0x162 => array(0x163), 0x164 => array(0x165),
                    0x166 => array(0x167), 0x168 => array(0x169), 0x16A => array(0x16B),
                    0x16C => array(0x16D), 0x16E => array(0x16F), 0x170 => array(0x171),
                    0x172 => array(0x173), 0x174 => array(0x175), 0x176 => array(0x177),
                    0x178 => array(0xFF), 0x179 => array(0x17A), 0x17B => array(0x17C),
                    0x17D => array(0x17E), 0x17F => array(0x73), 0x181 => array(0x253),
                    0x182 => array(0x183), 0x184 => array(0x185), 0x186 => array(0x254),
                    0x187 => array(0x188), 0x189 => array(0x256), 0x18A => array(0x257),
                    0x18B => array(0x18C), 0x18E => array(0x1DD), 0x18F => array(0x259),
                    0x190 => array(0x25B), 0x191 => array(0x192), 0x193 => array(0x260),
                    0x194 => array(0x263), 0x196 => array(0x269), 0x197 => array(0x268),
                    0x198 => array(0x199), 0x19C => array(0x26F), 0x19D => array(0x272),
                    0x19F => array(0x275), 0x1A0 => array(0x1A1), 0x1A2 => array(0x1A3),
                    0x1A4 => array(0x1A5), 0x1A6 => array(0x280), 0x1A7 => array(0x1A8),
                    0x1A9 => array(0x283), 0x1AC => array(0x1AD), 0x1AE => array(0x288),
                    0x1AF => array(0x1B0), 0x1B1 => array(0x28A), 0x1B2 => array(0x28B),
                    0x1B3 => array(0x1B4), 0x1B5 => array(0x1B6), 0x1B7 => array(0x292),
                    0x1B8 => array(0x1B9), 0x1BC => array(0x1BD), 0x1C4 => array(0x1C6),
                    0x1C5 => array(0x1C6), 0x1C7 => array(0x1C9), 0x1C8 => array(0x1C9),
                    0x1CA => array(0x1CC), 0x1CB => array(0x1CC), 0x1CD => array(0x1CE),
                    0x1CF => array(0x1D0), 0x1D1 => array(0x1D2), 0x1D3 => array(0x1D4),
                    0x1D5 => array(0x1D6), 0x1D7 => array(0x1D8), 0x1D9 => array(0x1DA),
                    0x1DB => array(0x1DC), 0x1DE => array(0x1DF), 0x1E0 => array(0x1E1),
                    0x1E2 => array(0x1E3), 0x1E4 => array(0x1E5), 0x1E6 => array(0x1E7),
                    0x1E8 => array(0x1E9), 0x1EA => array(0x1EB), 0x1EC => array(0x1ED),
                    0x1EE => array(0x1EF), 0x1F0 => array(0x6A, 0x30C), 0x1F1 => array(0x1F3),
                    0x1F2 => array(0x1F3), 0x1F4 => array(0x1F5), 0x1F6 => array(0x195),
                    0x1F7 => array(0x1BF), 0x1F8 => array(0x1F9), 0x1FA => array(0x1FB),
                    0x1FC => array(0x1FD), 0x1FE => array(0x1FF), 0x200 => array(0x201),
                    0x202 => array(0x203), 0x204 => array(0x205), 0x206 => array(0x207),
                    0x208 => array(0x209), 0x20A => array(0x20B), 0x20C => array(0x20D),
                    0x20E => array(0x20F), 0x210 => array(0x211), 0x212 => array(0x213),
                    0x214 => array(0x215), 0x216 => array(0x217), 0x218 => array(0x219),
                    0x21A => array(0x21B), 0x21C => array(0x21D), 0x21E => array(0x21F),
                    0x220 => array(0x19E), 0x222 => array(0x223), 0x224 => array(0x225),
                    0x226 => array(0x227), 0x228 => array(0x229), 0x22A => array(0x22B),
                    0x22C => array(0x22D), 0x22E => array(0x22F), 0x230 => array(0x231),
                    0x232 => array(0x233), 0x345 => array(0x3B9), 0x37A => array(0x20, 0x3B9),
                    0x386 => array(0x3AC), 0x388 => array(0x3AD), 0x389 => array(0x3AE),
                    0x38A => array(0x3AF), 0x38C => array(0x3CC), 0x38E => array(0x3CD),
                    0x38F => array(0x3CE), 0x390 => array(0x3B9, 0x308, 0x301),
                    0x391 => array(0x3B1), 0x392 => array(0x3B2), 0x393 => array(0x3B3),
                    0x394 => array(0x3B4), 0x395 => array(0x3B5), 0x396 => array(0x3B6),
                    0x397 => array(0x3B7), 0x398 => array(0x3B8), 0x399 => array(0x3B9),
                    0x39A => array(0x3BA), 0x39B => array(0x3BB), 0x39C => array(0x3BC),
                    0x39D => array(0x3BD), 0x39E => array(0x3BE), 0x39F => array(0x3BF),
                    0x3A0 => array(0x3C0), 0x3A1 => array(0x3C1), 0x3A3 => array(0x3C3),
                    0x3A4 => array(0x3C4), 0x3A5 => array(0x3C5), 0x3A6 => array(0x3C6),
                    0x3A7 => array(0x3C7), 0x3A8 => array(0x3C8), 0x3A9 => array(0x3C9),
                    0x3AA => array(0x3CA), 0x3AB => array(0x3CB), 0x3B0 => array(0x3C5, 0x308, 0x301),
                    0x3C2 => array(0x3C3), 0x3D0 => array(0x3B2), 0x3D1 => array(0x3B8),
                    0x3D2 => array(0x3C5), 0x3D3 => array(0x3CD), 0x3D4 => array(0x3CB),
                    0x3D5 => array(0x3C6), 0x3D6 => array(0x3C0), 0x3D8 => array(0x3D9),
                    0x3DA => array(0x3DB), 0x3DC => array(0x3DD), 0x3DE => array(0x3DF),
                    0x3E0 => array(0x3E1), 0x3E2 => array(0x3E3), 0x3E4 => array(0x3E5),
                    0x3E6 => array(0x3E7), 0x3E8 => array(0x3E9), 0x3EA => array(0x3EB),
                    0x3EC => array(0x3ED), 0x3EE => array(0x3EF), 0x3F0 => array(0x3BA),
                    0x3F1 => array(0x3C1), 0x3F2 => array(0x3C3), 0x3F4 => array(0x3B8),
                    0x3F5 => array(0x3B5), 0x400 => array(0x450), 0x401 => array(0x451),
                    0x402 => array(0x452), 0x403 => array(0x453), 0x404 => array(0x454),
                    0x405 => array(0x455), 0x406 => array(0x456), 0x407 => array(0x457),
                    0x408 => array(0x458), 0x409 => array(0x459), 0x40A => array(0x45A),
                    0x40B => array(0x45B), 0x40C => array(0x45C), 0x40D => array(0x45D),
                    0x40E => array(0x45E), 0x40F => array(0x45F), 0x410 => array(0x430),
                    0x411 => array(0x431), 0x412 => array(0x432), 0x413 => array(0x433),
                    0x414 => array(0x434), 0x415 => array(0x435), 0x416 => array(0x436),
                    0x417 => array(0x437), 0x418 => array(0x438), 0x419 => array(0x439),
                    0x41A => array(0x43A), 0x41B => array(0x43B), 0x41C => array(0x43C),
                    0x41D => array(0x43D), 0x41E => array(0x43E), 0x41F => array(0x43F),
                    0x420 => array(0x440), 0x421 => array(0x441), 0x422 => array(0x442),
                    0x423 => array(0x443), 0x424 => array(0x444), 0x425 => array(0x445),
                    0x426 => array(0x446), 0x427 => array(0x447), 0x428 => array(0x448),
                    0x429 => array(0x449), 0x42A => array(0x44A), 0x42B => array(0x44B),
                    0x42C => array(0x44C), 0x42D => array(0x44D), 0x42E => array(0x44E),
                    0x42F => array(0x44F), 0x460 => array(0x461), 0x462 => array(0x463),
                    0x464 => array(0x465), 0x466 => array(0x467), 0x468 => array(0x469),
                    0x46A => array(0x46B), 0x46C => array(0x46D), 0x46E => array(0x46F),
                    0x470 => array(0x471), 0x472 => array(0x473), 0x474 => array(0x475),
                    0x476 => array(0x477), 0x478 => array(0x479), 0x47A => array(0x47B),
                    0x47C => array(0x47D), 0x47E => array(0x47F), 0x480 => array(0x481),
                    0x48A => array(0x48B), 0x48C => array(0x48D), 0x48E => array(0x48F),
                    0x490 => array(0x491), 0x492 => array(0x493), 0x494 => array(0x495),
                    0x496 => array(0x497), 0x498 => array(0x499), 0x49A => array(0x49B),
                    0x49C => array(0x49D), 0x49E => array(0x49F), 0x4A0 => array(0x4A1),
                    0x4A2 => array(0x4A3), 0x4A4 => array(0x4A5), 0x4A6 => array(0x4A7),
                    0x4A8 => array(0x4A9), 0x4AA => array(0x4AB), 0x4AC => array(0x4AD),
                    0x4AE => array(0x4AF), 0x4B0 => array(0x4B1), 0x4B2 => array(0x4B3),
                    0x4B4 => array(0x4B5), 0x4B6 => array(0x4B7), 0x4B8 => array(0x4B9),
                    0x4BA => array(0x4BB), 0x4BC => array(0x4BD), 0x4BE => array(0x4BF),
                    0x4C1 => array(0x4C2), 0x4C3 => array(0x4C4), 0x4C5 => array(0x4C6),
                    0x4C7 => array(0x4C8), 0x4C9 => array(0x4CA), 0x4CB => array(0x4CC),
                    0x4CD => array(0x4CE), 0x4D0 => array(0x4D1), 0x4D2 => array(0x4D3),
                    0x4D4 => array(0x4D5), 0x4D6 => array(0x4D7), 0x4D8 => array(0x4D9),
                    0x4DA => array(0x4DB), 0x4DC => array(0x4DD), 0x4DE => array(0x4DF),
                    0x4E0 => array(0x4E1), 0x4E2 => array(0x4E3), 0x4E4 => array(0x4E5),
                    0x4E6 => array(0x4E7), 0x4E8 => array(0x4E9), 0x4EA => array(0x4EB),
                    0x4EC => array(0x4ED), 0x4EE => array(0x4EF), 0x4F0 => array(0x4F1),
                    0x4F2 => array(0x4F3), 0x4F4 => array(0x4F5), 0x4F8 => array(0x4F9),
                    0x500 => array(0x501), 0x502 => array(0x503), 0x504 => array(0x505),
                    0x506 => array(0x507), 0x508 => array(0x509), 0x50A => array(0x50B),
                    0x50C => array(0x50D), 0x50E => array(0x50F), 0x531 => array(0x561),
                    0x532 => array(0x562), 0x533 => array(0x563), 0x534 => array(0x564),
                    0x535 => array(0x565), 0x536 => array(0x566), 0x537 => array(0x567),
                    0x538 => array(0x568), 0x539 => array(0x569), 0x53A => array(0x56A),
                    0x53B => array(0x56B), 0x53C => array(0x56C), 0x53D => array(0x56D),
                    0x53E => array(0x56E), 0x53F => array(0x56F), 0x540 => array(0x570),
                    0x541 => array(0x571), 0x542 => array(0x572), 0x543 => array(0x573),
                    0x544 => array(0x574), 0x545 => array(0x575), 0x546 => array(0x576),
                    0x547 => array(0x577), 0x548 => array(0x578), 0x549 => array(0x579),
                    0x54A => array(0x57A), 0x54B => array(0x57B), 0x54C => array(0x57C),
                    0x54D => array(0x57D), 0x54E => array(0x57E), 0x54F => array(0x57F),
                    0x550 => array(0x580), 0x551 => array(0x581), 0x552 => array(0x582),
                    0x553 => array(0x583), 0x554 => array(0x584), 0x555 => array(0x585),
                    0x556 => array(0x586), 0x587 => array(0x565, 0x582), 0xE33 => array(0xE4D, 0xE32),
                    0x1E00 => array(0x1E01), 0x1E02 => array(0x1E03), 0x1E04 => array(0x1E05),
                    0x1E06 => array(0x1E07), 0x1E08 => array(0x1E09), 0x1E0A => array(0x1E0B),
                    0x1E0C => array(0x1E0D), 0x1E0E => array(0x1E0F), 0x1E10 => array(0x1E11),
                    0x1E12 => array(0x1E13), 0x1E14 => array(0x1E15), 0x1E16 => array(0x1E17),
                    0x1E18 => array(0x1E19), 0x1E1A => array(0x1E1B), 0x1E1C => array(0x1E1D),
                    0x1E1E => array(0x1E1F), 0x1E20 => array(0x1E21), 0x1E22 => array(0x1E23),
                    0x1E24 => array(0x1E25), 0x1E26 => array(0x1E27), 0x1E28 => array(0x1E29),
                    0x1E2A => array(0x1E2B), 0x1E2C => array(0x1E2D), 0x1E2E => array(0x1E2F),
                    0x1E30 => array(0x1E31), 0x1E32 => array(0x1E33), 0x1E34 => array(0x1E35),
                    0x1E36 => array(0x1E37), 0x1E38 => array(0x1E39), 0x1E3A => array(0x1E3B),
                    0x1E3C => array(0x1E3D), 0x1E3E => array(0x1E3F), 0x1E40 => array(0x1E41),
                    0x1E42 => array(0x1E43), 0x1E44 => array(0x1E45), 0x1E46 => array(0x1E47),
                    0x1E48 => array(0x1E49), 0x1E4A => array(0x1E4B), 0x1E4C => array(0x1E4D),
                    0x1E4E => array(0x1E4F), 0x1E50 => array(0x1E51), 0x1E52 => array(0x1E53),
                    0x1E54 => array(0x1E55), 0x1E56 => array(0x1E57), 0x1E58 => array(0x1E59),
                    0x1E5A => array(0x1E5B), 0x1E5C => array(0x1E5D), 0x1E5E => array(0x1E5F),
                    0x1E60 => array(0x1E61), 0x1E62 => array(0x1E63), 0x1E64 => array(0x1E65),
                    0x1E66 => array(0x1E67), 0x1E68 => array(0x1E69), 0x1E6A => array(0x1E6B),
                    0x1E6C => array(0x1E6D), 0x1E6E => array(0x1E6F), 0x1E70 => array(0x1E71),
                    0x1E72 => array(0x1E73), 0x1E74 => array(0x1E75), 0x1E76 => array(0x1E77),
                    0x1E78 => array(0x1E79), 0x1E7A => array(0x1E7B), 0x1E7C => array(0x1E7D),
                    0x1E7E => array(0x1E7F), 0x1E80 => array(0x1E81), 0x1E82 => array(0x1E83),
                    0x1E84 => array(0x1E85), 0x1E86 => array(0x1E87), 0x1E88 => array(0x1E89),
                    0x1E8A => array(0x1E8B), 0x1E8C => array(0x1E8D), 0x1E8E => array(0x1E8F),
                    0x1E90 => array(0x1E91), 0x1E92 => array(0x1E93), 0x1E94 => array(0x1E95),
                    0x1E96 => array(0x68, 0x331), 0x1E97 => array(0x74, 0x308), 0x1E98 => array(0x77, 0x30A),
                    0x1E99 => array(0x79, 0x30A), 0x1E9A => array(0x61, 0x2BE), 0x1E9B => array(0x1E61),
                    0x1EA0 => array(0x1EA1), 0x1EA2 => array(0x1EA3), 0x1EA4 => array(0x1EA5),
                    0x1EA6 => array(0x1EA7), 0x1EA8 => array(0x1EA9), 0x1EAA => array(0x1EAB),
                    0x1EAC => array(0x1EAD), 0x1EAE => array(0x1EAF), 0x1EB0 => array(0x1EB1),
                    0x1EB2 => array(0x1EB3), 0x1EB4 => array(0x1EB5), 0x1EB6 => array(0x1EB7),
                    0x1EB8 => array(0x1EB9), 0x1EBA => array(0x1EBB), 0x1EBC => array(0x1EBD),
                    0x1EBE => array(0x1EBF), 0x1EC0 => array(0x1EC1), 0x1EC2 => array(0x1EC3),
                    0x1EC4 => array(0x1EC5), 0x1EC6 => array(0x1EC7), 0x1EC8 => array(0x1EC9),
                    0x1ECA => array(0x1ECB), 0x1ECC => array(0x1ECD), 0x1ECE => array(0x1ECF),
                    0x1ED0 => array(0x1ED1), 0x1ED2 => array(0x1ED3), 0x1ED4 => array(0x1ED5),
                    0x1ED6 => array(0x1ED7), 0x1ED8 => array(0x1ED9), 0x1EDA => array(0x1EDB),
                    0x1EDC => array(0x1EDD), 0x1EDE => array(0x1EDF), 0x1EE0 => array(0x1EE1),
                    0x1EE2 => array(0x1EE3), 0x1EE4 => array(0x1EE5), 0x1EE6 => array(0x1EE7),
                    0x1EE8 => array(0x1EE9), 0x1EEA => array(0x1EEB), 0x1EEC => array(0x1EED),
                    0x1EEE => array(0x1EEF), 0x1EF0 => array(0x1EF1), 0x1EF2 => array(0x1EF3),
                    0x1EF4 => array(0x1EF5), 0x1EF6 => array(0x1EF7), 0x1EF8 => array(0x1EF9),
                    0x1F08 => array(0x1F00), 0x1F09 => array(0x1F01), 0x1F0A => array(0x1F02),
                    0x1F0B => array(0x1F03), 0x1F0C => array(0x1F04), 0x1F0D => array(0x1F05),
                    0x1F0E => array(0x1F06), 0x1F0F => array(0x1F07), 0x1F18 => array(0x1F10),
                    0x1F19 => array(0x1F11), 0x1F1A => array(0x1F12), 0x1F1B => array(0x1F13),
                    0x1F1C => array(0x1F14), 0x1F1D => array(0x1F15), 0x1F28 => array(0x1F20),
                    0x1F29 => array(0x1F21), 0x1F2A => array(0x1F22), 0x1F2B => array(0x1F23),
                    0x1F2C => array(0x1F24), 0x1F2D => array(0x1F25), 0x1F2E => array(0x1F26),
                    0x1F2F => array(0x1F27), 0x1F38 => array(0x1F30), 0x1F39 => array(0x1F31),
                    0x1F3A => array(0x1F32), 0x1F3B => array(0x1F33), 0x1F3C => array(0x1F34),
                    0x1F3D => array(0x1F35), 0x1F3E => array(0x1F36), 0x1F3F => array(0x1F37),
                    0x1F48 => array(0x1F40), 0x1F49 => array(0x1F41), 0x1F4A => array(0x1F42),
                    0x1F4B => array(0x1F43), 0x1F4C => array(0x1F44), 0x1F4D => array(0x1F45),
                    0x1F50 => array(0x3C5, 0x313), 0x1F52 => array(0x3C5, 0x313, 0x300),
                    0x1F54 => array(0x3C5, 0x313, 0x301), 0x1F56 => array(0x3C5, 0x313, 0x342),
                    0x1F59 => array(0x1F51), 0x1F5B => array(0x1F53), 0x1F5D => array(0x1F55),
                    0x1F5F => array(0x1F57), 0x1F68 => array(0x1F60), 0x1F69 => array(0x1F61),
                    0x1F6A => array(0x1F62), 0x1F6B => array(0x1F63), 0x1F6C => array(0x1F64),
                    0x1F6D => array(0x1F65), 0x1F6E => array(0x1F66), 0x1F6F => array(0x1F67),
                    0x1F80 => array(0x1F00, 0x3B9), 0x1F81 => array(0x1F01, 0x3B9),
                    0x1F82 => array(0x1F02, 0x3B9), 0x1F83 => array(0x1F03, 0x3B9),
                    0x1F84 => array(0x1F04, 0x3B9), 0x1F85 => array(0x1F05, 0x3B9),
                    0x1F86 => array(0x1F06, 0x3B9), 0x1F87 => array(0x1F07, 0x3B9),
                    0x1F88 => array(0x1F00, 0x3B9), 0x1F89 => array(0x1F01, 0x3B9),
                    0x1F8A => array(0x1F02, 0x3B9), 0x1F8B => array(0x1F03, 0x3B9),
                    0x1F8C => array(0x1F04, 0x3B9), 0x1F8D => array(0x1F05, 0x3B9),
                    0x1F8E => array(0x1F06, 0x3B9), 0x1F8F => array(0x1F07, 0x3B9),
                    0x1F90 => array(0x1F20, 0x3B9), 0x1F91 => array(0x1F21, 0x3B9),
                    0x1F92 => array(0x1F22, 0x3B9), 0x1F93 => array(0x1F23, 0x3B9),
                    0x1F94 => array(0x1F24, 0x3B9), 0x1F95 => array(0x1F25, 0x3B9),
                    0x1F96 => array(0x1F26, 0x3B9), 0x1F97 => array(0x1F27, 0x3B9),
                    0x1F98 => array(0x1F20, 0x3B9), 0x1F99 => array(0x1F21, 0x3B9),
                    0x1F9A => array(0x1F22, 0x3B9), 0x1F9B => array(0x1F23, 0x3B9),
                    0x1F9C => array(0x1F24, 0x3B9), 0x1F9D => array(0x1F25, 0x3B9),
                    0x1F9E => array(0x1F26, 0x3B9), 0x1F9F => array(0x1F27, 0x3B9),
                    0x1FA0 => array(0x1F60, 0x3B9), 0x1FA1 => array(0x1F61, 0x3B9),
                    0x1FA2 => array(0x1F62, 0x3B9), 0x1FA3 => array(0x1F63, 0x3B9),
                    0x1FA4 => array(0x1F64, 0x3B9), 0x1FA5 => array(0x1F65, 0x3B9),
                    0x1FA6 => array(0x1F66, 0x3B9), 0x1FA7 => array(0x1F67, 0x3B9),
                    0x1FA8 => array(0x1F60, 0x3B9), 0x1FA9 => array(0x1F61, 0x3B9),
                    0x1FAA => array(0x1F62, 0x3B9), 0x1FAB => array(0x1F63, 0x3B9),
                    0x1FAC => array(0x1F64, 0x3B9), 0x1FAD => array(0x1F65, 0x3B9),
                    0x1FAE => array(0x1F66, 0x3B9), 0x1FAF => array(0x1F67, 0x3B9),
                    0x1FB2 => array(0x1F70, 0x3B9), 0x1FB3 => array(0x3B1, 0x3B9),
                    0x1FB4 => array(0x3AC, 0x3B9), 0x1FB6 => array(0x3B1, 0x342),
                    0x1FB7 => array(0x3B1, 0x342, 0x3B9), 0x1FB8 => array(0x1FB0),
                    0x1FB9 => array(0x1FB1), 0x1FBA => array(0x1F70), 0x1FBB => array(0x1F71),
                    0x1FBC => array(0x3B1, 0x3B9), 0x1FBE => array(0x3B9),
                    0x1FC2 => array(0x1F74, 0x3B9), 0x1FC3 => array(0x3B7, 0x3B9),
                    0x1FC4 => array(0x3AE, 0x3B9), 0x1FC6 => array(0x3B7, 0x342),
                    0x1FC7 => array(0x3B7, 0x342, 0x3B9), 0x1FC8 => array(0x1F72),
                    0x1FC9 => array(0x1F73), 0x1FCA => array(0x1F74), 0x1FCB => array(0x1F75),
                    0x1FCC => array(0x3B7, 0x3B9), 0x1FD2 => array(0x3B9, 0x308, 0x300),
                    0x1FD3 => array(0x3B9, 0x308, 0x301), 0x1FD6 => array(0x3B9, 0x342),
                    0x1FD7 => array(0x3B9, 0x308, 0x342), 0x1FD8 => array(0x1FD0),
                    0x1FD9 => array(0x1FD1), 0x1FDA => array(0x1F76),
                    0x1FDB => array(0x1F77), 0x1FE2 => array(0x3C5, 0x308, 0x300),
                    0x1FE3 => array(0x3C5, 0x308, 0x301), 0x1FE4 => array(0x3C1, 0x313),
                    0x1FE6 => array(0x3C5, 0x342), 0x1FE7 => array(0x3C5, 0x308, 0x342),
                    0x1FE8 => array(0x1FE0), 0x1FE9 => array(0x1FE1),
                    0x1FEA => array(0x1F7A), 0x1FEB => array(0x1F7B),
                    0x1FEC => array(0x1FE5), 0x1FF2 => array(0x1F7C, 0x3B9),
                    0x1FF3 => array(0x3C9, 0x3B9), 0x1FF4 => array(0x3CE, 0x3B9),
                    0x1FF6 => array(0x3C9, 0x342), 0x1FF7 => array(0x3C9, 0x342, 0x3B9),
                    0x1FF8 => array(0x1F78), 0x1FF9 => array(0x1F79), 0x1FFA => array(0x1F7C),
                    0x1FFB => array(0x1F7D), 0x1FFC => array(0x3C9, 0x3B9),
                    0x20A8 => array(0x72, 0x73), 0x2102 => array(0x63), 0x2103 => array(0xB0, 0x63),
                    0x2107 => array(0x25B), 0x2109 => array(0xB0, 0x66), 0x210B => array(0x68),
                    0x210C => array(0x68), 0x210D => array(0x68), 0x2110 => array(0x69),
                    0x2111 => array(0x69), 0x2112 => array(0x6C), 0x2115 => array(0x6E),
                    0x2116 => array(0x6E, 0x6F), 0x2119 => array(0x70), 0x211A => array(0x71),
                    0x211B => array(0x72), 0x211C => array(0x72), 0x211D => array(0x72),
                    0x2120 => array(0x73, 0x6D), 0x2121 => array(0x74, 0x65, 0x6C),
                    0x2122 => array(0x74, 0x6D), 0x2124 => array(0x7A), 0x2126 => array(0x3C9),
                    0x2128 => array(0x7A), 0x212A => array(0x6B), 0x212B => array(0xE5),
                    0x212C => array(0x62), 0x212D => array(0x63), 0x2130 => array(0x65),
                    0x2131 => array(0x66), 0x2133 => array(0x6D), 0x213E => array(0x3B3),
                    0x213F => array(0x3C0), 0x2145 => array(0x64), 0x2160 => array(0x2170),
                    0x2161 => array(0x2171), 0x2162 => array(0x2172), 0x2163 => array(0x2173),
                    0x2164 => array(0x2174), 0x2165 => array(0x2175), 0x2166 => array(0x2176),
                    0x2167 => array(0x2177), 0x2168 => array(0x2178), 0x2169 => array(0x2179),
                    0x216A => array(0x217A), 0x216B => array(0x217B), 0x216C => array(0x217C),
                    0x216D => array(0x217D), 0x216E => array(0x217E), 0x216F => array(0x217F),
                    0x24B6 => array(0x24D0), 0x24B7 => array(0x24D1), 0x24B8 => array(0x24D2),
                    0x24B9 => array(0x24D3), 0x24BA => array(0x24D4), 0x24BB => array(0x24D5),
                    0x24BC => array(0x24D6), 0x24BD => array(0x24D7), 0x24BE => array(0x24D8),
                    0x24BF => array(0x24D9), 0x24C0 => array(0x24DA), 0x24C1 => array(0x24DB),
                    0x24C2 => array(0x24DC), 0x24C3 => array(0x24DD), 0x24C4 => array(0x24DE),
                    0x24C5 => array(0x24DF), 0x24C6 => array(0x24E0), 0x24C7 => array(0x24E1),
                    0x24C8 => array(0x24E2), 0x24C9 => array(0x24E3), 0x24CA => array(0x24E4),
                    0x24CB => array(0x24E5), 0x24CC => array(0x24E6), 0x24CD => array(0x24E7),
                    0x24CE => array(0x24E8), 0x24CF => array(0x24E9), 0x3371 => array(0x68, 0x70, 0x61),
                    0x3373 => array(0x61, 0x75), 0x3375 => array(0x6F, 0x76),
                    0x3380 => array(0x70, 0x61), 0x3381 => array(0x6E, 0x61),
                    0x3382 => array(0x3BC, 0x61), 0x3383 => array(0x6D, 0x61),
                    0x3384 => array(0x6B, 0x61), 0x3385 => array(0x6B, 0x62),
                    0x3386 => array(0x6D, 0x62), 0x3387 => array(0x67, 0x62),
                    0x338A => array(0x70, 0x66), 0x338B => array(0x6E, 0x66),
                    0x338C => array(0x3BC, 0x66), 0x3390 => array(0x68, 0x7A),
                    0x3391 => array(0x6B, 0x68, 0x7A), 0x3392 => array(0x6D, 0x68, 0x7A),
                    0x3393 => array(0x67, 0x68, 0x7A), 0x3394 => array(0x74, 0x68, 0x7A),
                    0x33A9 => array(0x70, 0x61), 0x33AA => array(0x6B, 0x70, 0x61),
                    0x33AB => array(0x6D, 0x70, 0x61), 0x33AC => array(0x67, 0x70, 0x61),
                    0x33B4 => array(0x70, 0x76), 0x33B5 => array(0x6E, 0x76),
                    0x33B6 => array(0x3BC, 0x76), 0x33B7 => array(0x6D, 0x76),
                    0x33B8 => array(0x6B, 0x76), 0x33B9 => array(0x6D, 0x76),
                    0x33BA => array(0x70, 0x77), 0x33BB => array(0x6E, 0x77),
                    0x33BC => array(0x3BC, 0x77), 0x33BD => array(0x6D, 0x77),
                    0x33BE => array(0x6B, 0x77), 0x33BF => array(0x6D, 0x77),
                    0x33C0 => array(0x6B, 0x3C9), 0x33C1 => array(0x6D, 0x3C9), /*
                    0x33C2  => array(0x61, 0x2E, 0x6D, 0x2E), */
                    0x33C3 => array(0x62, 0x71), 0x33C6 => array(0x63, 0x2215, 0x6B, 0x67),
                    0x33C7 => array(0x63, 0x6F, 0x2E), 0x33C8 => array(0x64, 0x62),
                    0x33C9 => array(0x67, 0x79), 0x33CB => array(0x68, 0x70),
                    0x33CD => array(0x6B, 0x6B), 0x33CE => array(0x6B, 0x6D),
                    0x33D7 => array(0x70, 0x68), 0x33D9 => array(0x70, 0x70, 0x6D),
                    0x33DA => array(0x70, 0x72), 0x33DC => array(0x73, 0x76),
                    0x33DD => array(0x77, 0x62), 0xFB00 => array(0x66, 0x66),
                    0xFB01 => array(0x66, 0x69), 0xFB02 => array(0x66, 0x6C),
                    0xFB03 => array(0x66, 0x66, 0x69), 0xFB04 => array(0x66, 0x66, 0x6C),
                    0xFB05 => array(0x73, 0x74), 0xFB06 => array(0x73, 0x74),
                    0xFB13 => array(0x574, 0x576), 0xFB14 => array(0x574, 0x565),
                    0xFB15 => array(0x574, 0x56B), 0xFB16 => array(0x57E, 0x576),
                    0xFB17 => array(0x574, 0x56D), 0xFF21 => array(0xFF41),
                    0xFF22 => array(0xFF42), 0xFF23 => array(0xFF43), 0xFF24 => array(0xFF44),
                    0xFF25 => array(0xFF45), 0xFF26 => array(0xFF46), 0xFF27 => array(0xFF47),
                    0xFF28 => array(0xFF48), 0xFF29 => array(0xFF49), 0xFF2A => array(0xFF4A),
                    0xFF2B => array(0xFF4B), 0xFF2C => array(0xFF4C), 0xFF2D => array(0xFF4D),
                    0xFF2E => array(0xFF4E), 0xFF2F => array(0xFF4F), 0xFF30 => array(0xFF50),
                    0xFF31 => array(0xFF51), 0xFF32 => array(0xFF52), 0xFF33 => array(0xFF53),
                    0xFF34 => array(0xFF54), 0xFF35 => array(0xFF55), 0xFF36 => array(0xFF56),
                    0xFF37 => array(0xFF57), 0xFF38 => array(0xFF58), 0xFF39 => array(0xFF59),
                    0xFF3A => array(0xFF5A), 0x10400 => array(0x10428), 0x10401 => array(0x10429),
                    0x10402 => array(0x1042A), 0x10403 => array(0x1042B), 0x10404 => array(0x1042C),
                    0x10405 => array(0x1042D), 0x10406 => array(0x1042E), 0x10407 => array(0x1042F),
                    0x10408 => array(0x10430), 0x10409 => array(0x10431), 0x1040A => array(0x10432),
                    0x1040B => array(0x10433), 0x1040C => array(0x10434), 0x1040D => array(0x10435),
                    0x1040E => array(0x10436), 0x1040F => array(0x10437), 0x10410 => array(0x10438),
                    0x10411 => array(0x10439), 0x10412 => array(0x1043A), 0x10413 => array(0x1043B),
                    0x10414 => array(0x1043C), 0x10415 => array(0x1043D), 0x10416 => array(0x1043E),
                    0x10417 => array(0x1043F), 0x10418 => array(0x10440), 0x10419 => array(0x10441),
                    0x1041A => array(0x10442), 0x1041B => array(0x10443), 0x1041C => array(0x10444),
                    0x1041D => array(0x10445), 0x1041E => array(0x10446), 0x1041F => array(0x10447),
                    0x10420 => array(0x10448), 0x10421 => array(0x10449), 0x10422 => array(0x1044A),
                    0x10423 => array(0x1044B), 0x10424 => array(0x1044C), 0x10425 => array(0x1044D),
                    0x1D400 => array(0x61), 0x1D401 => array(0x62), 0x1D402 => array(0x63),
                    0x1D403 => array(0x64), 0x1D404 => array(0x65), 0x1D405 => array(0x66),
                    0x1D406 => array(0x67), 0x1D407 => array(0x68), 0x1D408 => array(0x69),
                    0x1D409 => array(0x6A), 0x1D40A => array(0x6B), 0x1D40B => array(0x6C),
                    0x1D40C => array(0x6D), 0x1D40D => array(0x6E), 0x1D40E => array(0x6F),
                    0x1D40F => array(0x70), 0x1D410 => array(0x71), 0x1D411 => array(0x72),
                    0x1D412 => array(0x73), 0x1D413 => array(0x74), 0x1D414 => array(0x75),
                    0x1D415 => array(0x76), 0x1D416 => array(0x77), 0x1D417 => array(0x78),
                    0x1D418 => array(0x79), 0x1D419 => array(0x7A), 0x1D434 => array(0x61),
                    0x1D435 => array(0x62), 0x1D436 => array(0x63), 0x1D437 => array(0x64),
                    0x1D438 => array(0x65), 0x1D439 => array(0x66), 0x1D43A => array(0x67),
                    0x1D43B => array(0x68), 0x1D43C => array(0x69), 0x1D43D => array(0x6A),
                    0x1D43E => array(0x6B), 0x1D43F => array(0x6C), 0x1D440 => array(0x6D),
                    0x1D441 => array(0x6E), 0x1D442 => array(0x6F), 0x1D443 => array(0x70),
                    0x1D444 => array(0x71), 0x1D445 => array(0x72), 0x1D446 => array(0x73),
                    0x1D447 => array(0x74), 0x1D448 => array(0x75), 0x1D449 => array(0x76),
                    0x1D44A => array(0x77), 0x1D44B => array(0x78), 0x1D44C => array(0x79),
                    0x1D44D => array(0x7A), 0x1D468 => array(0x61), 0x1D469 => array(0x62),
                    0x1D46A => array(0x63), 0x1D46B => array(0x64), 0x1D46C => array(0x65),
                    0x1D46D => array(0x66), 0x1D46E => array(0x67), 0x1D46F => array(0x68),
                    0x1D470 => array(0x69), 0x1D471 => array(0x6A), 0x1D472 => array(0x6B),
                    0x1D473 => array(0x6C), 0x1D474 => array(0x6D), 0x1D475 => array(0x6E),
                    0x1D476 => array(0x6F), 0x1D477 => array(0x70), 0x1D478 => array(0x71),
                    0x1D479 => array(0x72), 0x1D47A => array(0x73), 0x1D47B => array(0x74),
                    0x1D47C => array(0x75), 0x1D47D => array(0x76), 0x1D47E => array(0x77),
                    0x1D47F => array(0x78), 0x1D480 => array(0x79), 0x1D481 => array(0x7A),
                    0x1D49C => array(0x61), 0x1D49E => array(0x63), 0x1D49F => array(0x64),
                    0x1D4A2 => array(0x67), 0x1D4A5 => array(0x6A), 0x1D4A6 => array(0x6B),
                    0x1D4A9 => array(0x6E), 0x1D4AA => array(0x6F), 0x1D4AB => array(0x70),
                    0x1D4AC => array(0x71), 0x1D4AE => array(0x73), 0x1D4AF => array(0x74),
                    0x1D4B0 => array(0x75), 0x1D4B1 => array(0x76), 0x1D4B2 => array(0x77),
                    0x1D4B3 => array(0x78), 0x1D4B4 => array(0x79), 0x1D4B5 => array(0x7A),
                    0x1D4D0 => array(0x61), 0x1D4D1 => array(0x62), 0x1D4D2 => array(0x63),
                    0x1D4D3 => array(0x64), 0x1D4D4 => array(0x65), 0x1D4D5 => array(0x66),
                    0x1D4D6 => array(0x67), 0x1D4D7 => array(0x68), 0x1D4D8 => array(0x69),
                    0x1D4D9 => array(0x6A), 0x1D4DA => array(0x6B), 0x1D4DB => array(0x6C),
                    0x1D4DC => array(0x6D), 0x1D4DD => array(0x6E), 0x1D4DE => array(0x6F),
                    0x1D4DF => array(0x70), 0x1D4E0 => array(0x71), 0x1D4E1 => array(0x72),
                    0x1D4E2 => array(0x73), 0x1D4E3 => array(0x74), 0x1D4E4 => array(0x75),
                    0x1D4E5 => array(0x76), 0x1D4E6 => array(0x77), 0x1D4E7 => array(0x78),
                    0x1D4E8 => array(0x79), 0x1D4E9 => array(0x7A), 0x1D504 => array(0x61),
                    0x1D505 => array(0x62), 0x1D507 => array(0x64), 0x1D508 => array(0x65),
                    0x1D509 => array(0x66), 0x1D50A => array(0x67), 0x1D50D => array(0x6A),
                    0x1D50E => array(0x6B), 0x1D50F => array(0x6C), 0x1D510 => array(0x6D),
                    0x1D511 => array(0x6E), 0x1D512 => array(0x6F), 0x1D513 => array(0x70),
                    0x1D514 => array(0x71), 0x1D516 => array(0x73), 0x1D517 => array(0x74),
                    0x1D518 => array(0x75), 0x1D519 => array(0x76), 0x1D51A => array(0x77),
                    0x1D51B => array(0x78), 0x1D51C => array(0x79), 0x1D538 => array(0x61),
                    0x1D539 => array(0x62), 0x1D53B => array(0x64), 0x1D53C => array(0x65),
                    0x1D53D => array(0x66), 0x1D53E => array(0x67), 0x1D540 => array(0x69),
                    0x1D541 => array(0x6A), 0x1D542 => array(0x6B), 0x1D543 => array(0x6C),
                    0x1D544 => array(0x6D), 0x1D546 => array(0x6F), 0x1D54A => array(0x73),
                    0x1D54B => array(0x74), 0x1D54C => array(0x75), 0x1D54D => array(0x76),
                    0x1D54E => array(0x77), 0x1D54F => array(0x78), 0x1D550 => array(0x79),
                    0x1D56C => array(0x61), 0x1D56D => array(0x62), 0x1D56E => array(0x63),
                    0x1D56F => array(0x64), 0x1D570 => array(0x65), 0x1D571 => array(0x66),
                    0x1D572 => array(0x67), 0x1D573 => array(0x68), 0x1D574 => array(0x69),
                    0x1D575 => array(0x6A), 0x1D576 => array(0x6B), 0x1D577 => array(0x6C),
                    0x1D578 => array(0x6D), 0x1D579 => array(0x6E), 0x1D57A => array(0x6F),
                    0x1D57B => array(0x70), 0x1D57C => array(0x71), 0x1D57D => array(0x72),
                    0x1D57E => array(0x73), 0x1D57F => array(0x74), 0x1D580 => array(0x75),
                    0x1D581 => array(0x76), 0x1D582 => array(0x77), 0x1D583 => array(0x78),
                    0x1D584 => array(0x79), 0x1D585 => array(0x7A), 0x1D5A0 => array(0x61),
                    0x1D5A1 => array(0x62), 0x1D5A2 => array(0x63), 0x1D5A3 => array(0x64),
                    0x1D5A4 => array(0x65), 0x1D5A5 => array(0x66), 0x1D5A6 => array(0x67),
                    0x1D5A7 => array(0x68), 0x1D5A8 => array(0x69), 0x1D5A9 => array(0x6A),
                    0x1D5AA => array(0x6B), 0x1D5AB => array(0x6C), 0x1D5AC => array(0x6D),
                    0x1D5AD => array(0x6E), 0x1D5AE => array(0x6F), 0x1D5AF => array(0x70),
                    0x1D5B0 => array(0x71), 0x1D5B1 => array(0x72), 0x1D5B2 => array(0x73),
                    0x1D5B3 => array(0x74), 0x1D5B4 => array(0x75), 0x1D5B5 => array(0x76),
                    0x1D5B6 => array(0x77), 0x1D5B7 => array(0x78), 0x1D5B8 => array(0x79),
                    0x1D5B9 => array(0x7A), 0x1D5D4 => array(0x61), 0x1D5D5 => array(0x62),
                    0x1D5D6 => array(0x63), 0x1D5D7 => array(0x64), 0x1D5D8 => array(0x65),
                    0x1D5D9 => array(0x66), 0x1D5DA => array(0x67), 0x1D5DB => array(0x68),
                    0x1D5DC => array(0x69), 0x1D5DD => array(0x6A), 0x1D5DE => array(0x6B),
                    0x1D5DF => array(0x6C), 0x1D5E0 => array(0x6D), 0x1D5E1 => array(0x6E),
                    0x1D5E2 => array(0x6F), 0x1D5E3 => array(0x70), 0x1D5E4 => array(0x71),
                    0x1D5E5 => array(0x72), 0x1D5E6 => array(0x73), 0x1D5E7 => array(0x74),
                    0x1D5E8 => array(0x75), 0x1D5E9 => array(0x76), 0x1D5EA => array(0x77),
                    0x1D5EB => array(0x78), 0x1D5EC => array(0x79), 0x1D5ED => array(0x7A),
                    0x1D608 => array(0x61), 0x1D609 => array(0x62), 0x1D60A => array(0x63),
                    0x1D60B => array(0x64), 0x1D60C => array(0x65), 0x1D60D => array(0x66),
                    0x1D60E => array(0x67), 0x1D60F => array(0x68), 0x1D610 => array(0x69),
                    0x1D611 => array(0x6A), 0x1D612 => array(0x6B), 0x1D613 => array(0x6C),
                    0x1D614 => array(0x6D), 0x1D615 => array(0x6E), 0x1D616 => array(0x6F),
                    0x1D617 => array(0x70), 0x1D618 => array(0x71), 0x1D619 => array(0x72),
                    0x1D61A => array(0x73), 0x1D61B => array(0x74), 0x1D61C => array(0x75),
                    0x1D61D => array(0x76), 0x1D61E => array(0x77), 0x1D61F => array(0x78),
                    0x1D620 => array(0x79), 0x1D621 => array(0x7A), 0x1D63C => array(0x61),
                    0x1D63D => array(0x62), 0x1D63E => array(0x63), 0x1D63F => array(0x64),
                    0x1D640 => array(0x65), 0x1D641 => array(0x66), 0x1D642 => array(0x67),
                    0x1D643 => array(0x68), 0x1D644 => array(0x69), 0x1D645 => array(0x6A),
                    0x1D646 => array(0x6B), 0x1D647 => array(0x6C), 0x1D648 => array(0x6D),
                    0x1D649 => array(0x6E), 0x1D64A => array(0x6F), 0x1D64B => array(0x70),
                    0x1D64C => array(0x71), 0x1D64D => array(0x72), 0x1D64E => array(0x73),
                    0x1D64F => array(0x74), 0x1D650 => array(0x75), 0x1D651 => array(0x76),
                    0x1D652 => array(0x77), 0x1D653 => array(0x78), 0x1D654 => array(0x79),
                    0x1D655 => array(0x7A), 0x1D670 => array(0x61), 0x1D671 => array(0x62),
                    0x1D672 => array(0x63), 0x1D673 => array(0x64), 0x1D674 => array(0x65),
                    0x1D675 => array(0x66), 0x1D676 => array(0x67), 0x1D677 => array(0x68),
                    0x1D678 => array(0x69), 0x1D679 => array(0x6A), 0x1D67A => array(0x6B),
                    0x1D67B => array(0x6C), 0x1D67C => array(0x6D), 0x1D67D => array(0x6E),
                    0x1D67E => array(0x6F), 0x1D67F => array(0x70), 0x1D680 => array(0x71),
                    0x1D681 => array(0x72), 0x1D682 => array(0x73), 0x1D683 => array(0x74),
                    0x1D684 => array(0x75), 0x1D685 => array(0x76), 0x1D686 => array(0x77),
                    0x1D687 => array(0x78), 0x1D688 => array(0x79), 0x1D689 => array(0x7A),
                    0x1D6A8 => array(0x3B1), 0x1D6A9 => array(0x3B2), 0x1D6AA => array(0x3B3),
                    0x1D6AB => array(0x3B4), 0x1D6AC => array(0x3B5), 0x1D6AD => array(0x3B6),
                    0x1D6AE => array(0x3B7), 0x1D6AF => array(0x3B8), 0x1D6B0 => array(0x3B9),
                    0x1D6B1 => array(0x3BA), 0x1D6B2 => array(0x3BB), 0x1D6B3 => array(0x3BC),
                    0x1D6B4 => array(0x3BD), 0x1D6B5 => array(0x3BE), 0x1D6B6 => array(0x3BF),
                    0x1D6B7 => array(0x3C0), 0x1D6B8 => array(0x3C1), 0x1D6B9 => array(0x3B8),
                    0x1D6BA => array(0x3C3), 0x1D6BB => array(0x3C4), 0x1D6BC => array(0x3C5),
                    0x1D6BD => array(0x3C6), 0x1D6BE => array(0x3C7), 0x1D6BF => array(0x3C8),
                    0x1D6C0 => array(0x3C9), 0x1D6D3 => array(0x3C3), 0x1D6E2 => array(0x3B1),
                    0x1D6E3 => array(0x3B2), 0x1D6E4 => array(0x3B3), 0x1D6E5 => array(0x3B4),
                    0x1D6E6 => array(0x3B5), 0x1D6E7 => array(0x3B6), 0x1D6E8 => array(0x3B7),
                    0x1D6E9 => array(0x3B8), 0x1D6EA => array(0x3B9), 0x1D6EB => array(0x3BA),
                    0x1D6EC => array(0x3BB), 0x1D6ED => array(0x3BC), 0x1D6EE => array(0x3BD),
                    0x1D6EF => array(0x3BE), 0x1D6F0 => array(0x3BF), 0x1D6F1 => array(0x3C0),
                    0x1D6F2 => array(0x3C1), 0x1D6F3 => array(0x3B8), 0x1D6F4 => array(0x3C3),
                    0x1D6F5 => array(0x3C4), 0x1D6F6 => array(0x3C5), 0x1D6F7 => array(0x3C6),
                    0x1D6F8 => array(0x3C7), 0x1D6F9 => array(0x3C8), 0x1D6FA => array(0x3C9),
                    0x1D70D => array(0x3C3), 0x1D71C => array(0x3B1), 0x1D71D => array(0x3B2),
                    0x1D71E => array(0x3B3), 0x1D71F => array(0x3B4), 0x1D720 => array(0x3B5),
                    0x1D721 => array(0x3B6), 0x1D722 => array(0x3B7), 0x1D723 => array(0x3B8),
                    0x1D724 => array(0x3B9), 0x1D725 => array(0x3BA), 0x1D726 => array(0x3BB),
                    0x1D727 => array(0x3BC), 0x1D728 => array(0x3BD), 0x1D729 => array(0x3BE),
                    0x1D72A => array(0x3BF), 0x1D72B => array(0x3C0), 0x1D72C => array(0x3C1),
                    0x1D72D => array(0x3B8), 0x1D72E => array(0x3C3), 0x1D72F => array(0x3C4),
                    0x1D730 => array(0x3C5), 0x1D731 => array(0x3C6), 0x1D732 => array(0x3C7),
                    0x1D733 => array(0x3C8), 0x1D734 => array(0x3C9), 0x1D747 => array(0x3C3),
                    0x1D756 => array(0x3B1), 0x1D757 => array(0x3B2), 0x1D758 => array(0x3B3),
                    0x1D759 => array(0x3B4), 0x1D75A => array(0x3B5), 0x1D75B => array(0x3B6),
                    0x1D75C => array(0x3B7), 0x1D75D => array(0x3B8), 0x1D75E => array(0x3B9),
                    0x1D75F => array(0x3BA), 0x1D760 => array(0x3BB), 0x1D761 => array(0x3BC),
                    0x1D762 => array(0x3BD), 0x1D763 => array(0x3BE), 0x1D764 => array(0x3BF),
                    0x1D765 => array(0x3C0), 0x1D766 => array(0x3C1), 0x1D767 => array(0x3B8),
                    0x1D768 => array(0x3C3), 0x1D769 => array(0x3C4), 0x1D76A => array(0x3C5),
                    0x1D76B => array(0x3C6), 0x1D76C => array(0x3C7), 0x1D76D => array(0x3C8),
                    0x1D76E => array(0x3C9), 0x1D781 => array(0x3C3), 0x1D790 => array(0x3B1),
                    0x1D791 => array(0x3B2), 0x1D792 => array(0x3B3), 0x1D793 => array(0x3B4),
                    0x1D794 => array(0x3B5), 0x1D795 => array(0x3B6), 0x1D796 => array(0x3B7),
                    0x1D797 => array(0x3B8), 0x1D798 => array(0x3B9), 0x1D799 => array(0x3BA),
                    0x1D79A => array(0x3BB), 0x1D79B => array(0x3BC), 0x1D79C => array(0x3BD),
                    0x1D79D => array(0x3BE), 0x1D79E => array(0x3BF), 0x1D79F => array(0x3C0),
                    0x1D7A0 => array(0x3C1), 0x1D7A1 => array(0x3B8), 0x1D7A2 => array(0x3C3),
                    0x1D7A3 => array(0x3C4), 0x1D7A4 => array(0x3C5), 0x1D7A5 => array(0x3C6),
                    0x1D7A6 => array(0x3C7), 0x1D7A7 => array(0x3C8), 0x1D7A8 => array(0x3C9),
                    0x1D7BB => array(0x3C3), 0x3F9 => array(0x3C3), 0x1D2C => array(0x61),
                    0x1D2D => array(0xE6), 0x1D2E => array(0x62), 0x1D30 => array(0x64),
                    0x1D31 => array(0x65), 0x1D32 => array(0x1DD), 0x1D33 => array(0x67),
                    0x1D34 => array(0x68), 0x1D35 => array(0x69), 0x1D36 => array(0x6A),
                    0x1D37 => array(0x6B), 0x1D38 => array(0x6C), 0x1D39 => array(0x6D),
                    0x1D3A => array(0x6E), 0x1D3C => array(0x6F), 0x1D3D => array(0x223),
                    0x1D3E => array(0x70), 0x1D3F => array(0x72), 0x1D40 => array(0x74),
                    0x1D41 => array(0x75), 0x1D42 => array(0x77), 0x213B => array(0x66, 0x61, 0x78),
                    0x3250 => array(0x70, 0x74, 0x65), 0x32CC => array(0x68, 0x67),
                    0x32CE => array(0x65, 0x76), 0x32CF => array(0x6C, 0x74, 0x64),
                    0x337A => array(0x69, 0x75), 0x33DE => array(0x76, 0x2215, 0x6D),
                    0x33DF => array(0x61, 0x2215, 0x6D)
            ),
            'replacemaps' => array(0x41 => array(0x61), 0x42 => array(0x62), 0x43 => array(0x63),
                    0x44 => array(0x64), 0x45 => array(0x65), 0x46 => array(0x66),
                    0x47 => array(0x67), 0x48 => array(0x68), 0x49 => array(0x69),
                    0x4A => array(0x6A), 0x4B => array(0x6B), 0x4C => array(0x6C),
                    0x4D => array(0x6D), 0x4E => array(0x6E), 0x4F => array(0x6F),
                    0x50 => array(0x70), 0x51 => array(0x71), 0x52 => array(0x72),
                    0x53 => array(0x73), 0x54 => array(0x74), 0x55 => array(0x75),
                    0x56 => array(0x76), 0x57 => array(0x77), 0x58 => array(0x78),
                    0x59 => array(0x79), 0x5A => array(0x7A), 0xAA => array(0x61),
                    0xB2 => array(0x32), 0xB3 => array(0x33), 0xB5 => array(0x3BC),
                    0xB9 => array(0x31), 0xBA => array(0x6F), 0xBC => array(0x31, 0x2044, 0x34),
                    0xBD => array(0x31, 0x2044, 0x32), 0xBE => array(0x33, 0x2044, 0x34), 0xC0 => array(0xE0),
                    0xC1 => array(0xE1), 0xC2 => array(0xE2), 0xC3 => array(0xE3),
                    0xC4 => array(0xE4), 0xC5 => array(0xE5), 0xC6 => array(0xE6),
                    0xC7 => array(0xE7), 0xC8 => array(0xE8), 0xC9 => array(0xE9),
                    0xCA => array(0xEA), 0xCB => array(0xEB), 0xCC => array(0xEC),
                    0xCD => array(0xED), 0xCE => array(0xEE), 0xCF => array(0xEF),
                    0xD0 => array(0xF0), 0xD1 => array(0xF1), 0xD2 => array(0xF2),
                    0xD3 => array(0xF3), 0xD4 => array(0xF4), 0xD5 => array(0xF5),
                    0xD6 => array(0xF6), 0xD8 => array(0xF8), 0xD9 => array(0xF9),
                    0xDA => array(0xFA), 0xDB => array(0xFB), 0xDC => array(0xFC),
                    0xDD => array(0xFD), 0xDE => array(0xFE), 0x100 => array(0x101),
                    0x102 => array(0x103), 0x104 => array(0x105), 0x106 => array(0x107),
                    0x108 => array(0x109), 0x10A => array(0x10B), 0x10C => array(0x10D),
                    0x10E => array(0x10F), 0x110 => array(0x111), 0x112 => array(0x113),
                    0x114 => array(0x115), 0x116 => array(0x117), 0x118 => array(0x119),
                    0x11A => array(0x11B), 0x11C => array(0x11D), 0x11E => array(0x11F),
                    0x120 => array(0x121), 0x122 => array(0x123), 0x124 => array(0x125),
                    0x126 => array(0x127), 0x128 => array(0x129), 0x12A => array(0x12B),
                    0x12C => array(0x12D), 0x12E => array(0x12F), 0x130 => array(0x69, 0x307),
                    0x132 => array(0x69, 0x6A), 0x133 => array(0x69, 0x6A), 0x134 => array(0x135),
                    0x136 => array(0x137), 0x139 => array(0x13A), 0x13B => array(0x13C),
                    0x13D => array(0x13E), 0x13F => array(0x6C, 0xB7), 0x140 => array(0x6C, 0xB7),
                    0x141 => array(0x142), 0x143 => array(0x144), 0x145 => array(0x146),
                    0x147 => array(0x148), 0x149 => array(0x2BC, 0x6E), 0x14A => array(0x14B),
                    0x14C => array(0x14D), 0x14E => array(0x14F), 0x150 => array(0x151),
                    0x152 => array(0x153), 0x154 => array(0x155), 0x156 => array(0x157),
                    0x158 => array(0x159), 0x15A => array(0x15B), 0x15C => array(0x15D),
                    0x15E => array(0x15F), 0x160 => array(0x161), 0x162 => array(0x163),
                    0x164 => array(0x165), 0x166 => array(0x167), 0x168 => array(0x169),
                    0x16A => array(0x16B), 0x16C => array(0x16D), 0x16E => array(0x16F),
                    0x170 => array(0x171), 0x172 => array(0x173), 0x174 => array(0x175),
                    0x176 => array(0x177), 0x178 => array(0xFF), 0x179 => array(0x17A),
                    0x17B => array(0x17C), 0x17D => array(0x17E), 0x17F => array(0x73),
                    0x181 => array(0x253), 0x182 => array(0x183), 0x184 => array(0x185),
                    0x186 => array(0x254), 0x187 => array(0x188), 0x189 => array(0x256),
                    0x18A => array(0x257), 0x18B => array(0x18C), 0x18E => array(0x1DD),
                    0x18F => array(0x259), 0x190 => array(0x25B), 0x191 => array(0x192),
                    0x193 => array(0x260), 0x194 => array(0x263), 0x196 => array(0x269),
                    0x197 => array(0x268), 0x198 => array(0x199), 0x19C => array(0x26F),
                    0x19D => array(0x272), 0x19F => array(0x275), 0x1A0 => array(0x1A1),
                    0x1A2 => array(0x1A3), 0x1A4 => array(0x1A5), 0x1A6 => array(0x280),
                    0x1A7 => array(0x1A8), 0x1A9 => array(0x283), 0x1AC => array(0x1AD),
                    0x1AE => array(0x288), 0x1AF => array(0x1B0), 0x1B1 => array(0x28A),
                    0x1B2 => array(0x28B), 0x1B3 => array(0x1B4), 0x1B5 => array(0x1B6),
                    0x1B7 => array(0x292), 0x1B8 => array(0x1B9), 0x1BC => array(0x1BD),
                    0x1C4 => array(0x64, 0x17E), 0x1C5 => array(0x64, 0x17E), 0x1C6 => array(0x64, 0x17E),
                    0x1C7 => array(0x6C, 0x6A), 0x1C8 => array(0x6C, 0x6A), 0x1C9 => array(0x6C, 0x6A),
                    0x1CA => array(0x6E, 0x6A), 0x1CB => array(0x6E, 0x6A), 0x1CC => array(0x6E, 0x6A),
                    0x1CD => array(0x1CE), 0x1CF => array(0x1D0), 0x1D1 => array(0x1D2),
                    0x1D3 => array(0x1D4), 0x1D5 => array(0x1D6), 0x1D7 => array(0x1D8),
                    0x1D9 => array(0x1DA), 0x1DB => array(0x1DC), 0x1DE => array(0x1DF),
                    0x1E0 => array(0x1E1), 0x1E2 => array(0x1E3), 0x1E4 => array(0x1E5),
                    0x1E6 => array(0x1E7), 0x1E8 => array(0x1E9), 0x1EA => array(0x1EB),
                    0x1EC => array(0x1ED), 0x1EE => array(0x1EF), 0x1F1 => array(0x64, 0x7A),
                    0x1F2 => array(0x64, 0x7A), 0x1F3 => array(0x64, 0x7A), 0x1F4 => array(0x1F5),
                    0x1F6 => array(0x195), 0x1F7 => array(0x1BF), 0x1F8 => array(0x1F9),
                    0x1FA => array(0x1FB), 0x1FC => array(0x1FD), 0x1FE => array(0x1FF),
                    0x200 => array(0x201), 0x202 => array(0x203), 0x204 => array(0x205),
                    0x206 => array(0x207), 0x208 => array(0x209), 0x20A => array(0x20B),
                    0x20C => array(0x20D), 0x20E => array(0x20F), 0x210 => array(0x211),
                    0x212 => array(0x213), 0x214 => array(0x215), 0x216 => array(0x217),
                    0x218 => array(0x219), 0x21A => array(0x21B), 0x21C => array(0x21D),
                    0x21E => array(0x21F), 0x220 => array(0x19E), 0x222 => array(0x223),
                    0x224 => array(0x225), 0x226 => array(0x227), 0x228 => array(0x229),
                    0x22A => array(0x22B), 0x22C => array(0x22D), 0x22E => array(0x22F),
                    0x230 => array(0x231), 0x232 => array(0x233), 0x23A => array(0x2C65),
                    0x23B => array(0x23C), 0x23D => array(0x19A), 0x23E => array(0x2C66),
                    0x241 => array(0x242), 0x243 => array(0x180), 0x244 => array(0x289),
                    0x245 => array(0x28C), 0x246 => array(0x247), 0x248 => array(0x249),
                    0x24A => array(0x24B), 0x24C => array(0x24D), 0x24E => array(0x24F),
                    0x2B0 => array(0x68), 0x2B1 => array(0x266), 0x2B2 => array(0x6A),
                    0x2B3 => array(0x72), 0x2B4 => array(0x279), 0x2B5 => array(0x27B),
                    0x2B6 => array(0x281), 0x2B7 => array(0x77), 0x2B8 => array(0x79),
                    0x2E0 => array(0x263), 0x2E1 => array(0x6C), 0x2E2 => array(0x73),
                    0x2E3 => array(0x78), 0x2E4 => array(0x295), 0x340 => array(0x300),
                    0x341 => array(0x301), 0x343 => array(0x313), 0x344 => array(0x308, 0x301),
                    0x345 => array(0x3B9), 0x370 => array(0x371), 0x372 => array(0x373),
                    0x374 => array(0x2B9), 0x376 => array(0x377), 0x37F => array(0x3F3),
                    0x386 => array(0x3AC), 0x387 => array(0xB7), 0x388 => array(0x3AD),
                    0x389 => array(0x3AE), 0x38A => array(0x3AF), 0x38C => array(0x3CC),
                    0x38E => array(0x3CD), 0x38F => array(0x3CE), 0x391 => array(0x3B1),
                    0x392 => array(0x3B2), 0x393 => array(0x3B3), 0x394 => array(0x3B4),
                    0x395 => array(0x3B5), 0x396 => array(0x3B6), 0x397 => array(0x3B7),
                    0x398 => array(0x3B8), 0x399 => array(0x3B9), 0x39A => array(0x3BA),
                    0x39B => array(0x3BB), 0x39C => array(0x3BC), 0x39D => array(0x3BD),
                    0x39E => array(0x3BE), 0x39F => array(0x3BF), 0x3A0 => array(0x3C0),
                    0x3A1 => array(0x3C1), 0x3A3 => array(0x3C3), 0x3A4 => array(0x3C4),
                    0x3A5 => array(0x3C5), 0x3A6 => array(0x3C6), 0x3A7 => array(0x3C7),
                    0x3A8 => array(0x3C8), 0x3A9 => array(0x3C9), 0x3AA => array(0x3CA),
                    0x3AB => array(0x3CB), 0x3CF => array(0x3D7), 0x3D0 => array(0x3B2),
                    0x3D1 => array(0x3B8), 0x3D2 => array(0x3C5), 0x3D3 => array(0x3CD),
                    0x3D4 => array(0x3CB), 0x3D5 => array(0x3C6), 0x3D6 => array(0x3C0),
                    0x3D8 => array(0x3D9), 0x3DA => array(0x3DB), 0x3DC => array(0x3DD),
                    0x3DE => array(0x3DF), 0x3E0 => array(0x3E1), 0x3E2 => array(0x3E3),
                    0x3E4 => array(0x3E5), 0x3E6 => array(0x3E7), 0x3E8 => array(0x3E9),
                    0x3EA => array(0x3EB), 0x3EC => array(0x3ED), 0x3EE => array(0x3EF),
                    0x3F0 => array(0x3BA), 0x3F1 => array(0x3C1), 0x3F2 => array(0x3C3),
                    0x3F4 => array(0x3B8), 0x3F5 => array(0x3B5), 0x3F7 => array(0x3F8),
                    0x3F9 => array(0x3C3), 0x3FA => array(0x3FB), 0x3FD => array(0x37B),
                    0x3FE => array(0x37C), 0x3FF => array(0x37D), 0x400 => array(0x450),
                    0x401 => array(0x451), 0x402 => array(0x452), 0x403 => array(0x453),
                    0x404 => array(0x454), 0x405 => array(0x455), 0x406 => array(0x456),
                    0x407 => array(0x457), 0x408 => array(0x458), 0x409 => array(0x459),
                    0x40A => array(0x45A), 0x40B => array(0x45B), 0x40C => array(0x45C),
                    0x40D => array(0x45D), 0x40E => array(0x45E), 0x40F => array(0x45F),
                    0x410 => array(0x430), 0x411 => array(0x431), 0x412 => array(0x432),
                    0x413 => array(0x433), 0x414 => array(0x434), 0x415 => array(0x435),
                    0x416 => array(0x436), 0x417 => array(0x437), 0x418 => array(0x438),
                    0x419 => array(0x439), 0x41A => array(0x43A), 0x41B => array(0x43B),
                    0x41C => array(0x43C), 0x41D => array(0x43D), 0x41E => array(0x43E),
                    0x41F => array(0x43F), 0x420 => array(0x440), 0x421 => array(0x441),
                    0x422 => array(0x442), 0x423 => array(0x443), 0x424 => array(0x444),
                    0x425 => array(0x445), 0x426 => array(0x446), 0x427 => array(0x447),
                    0x428 => array(0x448), 0x429 => array(0x449), 0x42A => array(0x44A),
                    0x42B => array(0x44B), 0x42C => array(0x44C), 0x42D => array(0x44D),
                    0x42E => array(0x44E), 0x42F => array(0x44F), 0x460 => array(0x461),
                    0x462 => array(0x463), 0x464 => array(0x465), 0x466 => array(0x467),
                    0x468 => array(0x469), 0x46A => array(0x46B), 0x46C => array(0x46D),
                    0x46E => array(0x46F), 0x470 => array(0x471), 0x472 => array(0x473),
                    0x474 => array(0x475), 0x476 => array(0x477), 0x478 => array(0x479),
                    0x47A => array(0x47B), 0x47C => array(0x47D), 0x47E => array(0x47F),
                    0x480 => array(0x481), 0x48A => array(0x48B), 0x48C => array(0x48D),
                    0x48E => array(0x48F), 0x490 => array(0x491), 0x492 => array(0x493),
                    0x494 => array(0x495), 0x496 => array(0x497), 0x498 => array(0x499),
                    0x49A => array(0x49B), 0x49C => array(0x49D), 0x49E => array(0x49F),
                    0x4A0 => array(0x4A1), 0x4A2 => array(0x4A3), 0x4A4 => array(0x4A5),
                    0x4A6 => array(0x4A7), 0x4A8 => array(0x4A9), 0x4AA => array(0x4AB),
                    0x4AC => array(0x4AD), 0x4AE => array(0x4AF), 0x4B0 => array(0x4B1),
                    0x4B2 => array(0x4B3), 0x4B4 => array(0x4B5), 0x4B6 => array(0x4B7),
                    0x4B8 => array(0x4B9), 0x4BA => array(0x4BB), 0x4BC => array(0x4BD),
                    0x4BE => array(0x4BF), 0x4C1 => array(0x4C2), 0x4C3 => array(0x4C4),
                    0x4C5 => array(0x4C6), 0x4C7 => array(0x4C8), 0x4C9 => array(0x4CA),
                    0x4CB => array(0x4CC), 0x4CD => array(0x4CE), 0x4D0 => array(0x4D1),
                    0x4D2 => array(0x4D3), 0x4D4 => array(0x4D5), 0x4D6 => array(0x4D7),
                    0x4D8 => array(0x4D9), 0x4DA => array(0x4DB), 0x4DC => array(0x4DD),
                    0x4DE => array(0x4DF), 0x4E0 => array(0x4E1), 0x4E2 => array(0x4E3),
                    0x4E4 => array(0x4E5), 0x4E6 => array(0x4E7), 0x4E8 => array(0x4E9),
                    0x4EA => array(0x4EB), 0x4EC => array(0x4ED), 0x4EE => array(0x4EF),
                    0x4F0 => array(0x4F1), 0x4F2 => array(0x4F3), 0x4F4 => array(0x4F5),
                    0x4F6 => array(0x4F7), 0x4F8 => array(0x4F9), 0x4FA => array(0x4FB),
                    0x4FC => array(0x4FD), 0x4FE => array(0x4FF), 0x500 => array(0x501),
                    0x502 => array(0x503), 0x504 => array(0x505), 0x506 => array(0x507),
                    0x508 => array(0x509), 0x50A => array(0x50B), 0x50C => array(0x50D),
                    0x50E => array(0x50F), 0x510 => array(0x511), 0x512 => array(0x513),
                    0x514 => array(0x515), 0x516 => array(0x517), 0x518 => array(0x519),
                    0x51A => array(0x51B), 0x51C => array(0x51D), 0x51E => array(0x51F),
                    0x520 => array(0x521), 0x522 => array(0x523), 0x524 => array(0x525),
                    0x526 => array(0x527), 0x528 => array(0x529), 0x52A => array(0x52B),
                    0x52C => array(0x52D), 0x52E => array(0x52F), 0x531 => array(0x561),
                    0x532 => array(0x562), 0x533 => array(0x563), 0x534 => array(0x564),
                    0x535 => array(0x565), 0x536 => array(0x566), 0x537 => array(0x567),
                    0x538 => array(0x568), 0x539 => array(0x569), 0x53A => array(0x56A),
                    0x53B => array(0x56B), 0x53C => array(0x56C), 0x53D => array(0x56D),
                    0x53E => array(0x56E), 0x53F => array(0x56F), 0x540 => array(0x570),
                    0x541 => array(0x571), 0x542 => array(0x572), 0x543 => array(0x573),
                    0x544 => array(0x574), 0x545 => array(0x575), 0x546 => array(0x576),
                    0x547 => array(0x577), 0x548 => array(0x578), 0x549 => array(0x579),
                    0x54A => array(0x57A), 0x54B => array(0x57B), 0x54C => array(0x57C),
                    0x54D => array(0x57D), 0x54E => array(0x57E), 0x54F => array(0x57F),
                    0x550 => array(0x580), 0x551 => array(0x581), 0x552 => array(0x582),
                    0x553 => array(0x583), 0x554 => array(0x584), 0x555 => array(0x585),
                    0x556 => array(0x586), 0x587 => array(0x565, 0x582), 0x675 => array(0x627, 0x674),
                    0x676 => array(0x648, 0x674), 0x677 => array(0x6C7, 0x674), 0x678 => array(0x64A, 0x674),
                    0x958 => array(0x915, 0x93C), 0x959 => array(0x916, 0x93C), 0x95A => array(0x917, 0x93C),
                    0x95B => array(0x91C, 0x93C), 0x95C => array(0x921, 0x93C), 0x95D => array(0x922, 0x93C),
                    0x95E => array(0x92B, 0x93C), 0x95F => array(0x92F, 0x93C), 0x9DC => array(0x9A1, 0x9BC),
                    0x9DD => array(0x9A2, 0x9BC), 0x9DF => array(0x9AF, 0x9BC), 0xA33 => array(0xA32, 0xA3C),
                    0xA36 => array(0xA38, 0xA3C), 0xA59 => array(0xA16, 0xA3C), 0xA5A => array(0xA17, 0xA3C),
                    0xA5B => array(0xA1C, 0xA3C), 0xA5E => array(0xA2B, 0xA3C), 0xB5C => array(0xB21, 0xB3C),
                    0xB5D => array(0xB22, 0xB3C), 0xE33 => array(0xE4D, 0xE32), 0xEB3 => array(0xECD, 0xEB2),
                    0xEDC => array(0xEAB, 0xE99), 0xEDD => array(0xEAB, 0xEA1), 0xF0C => array(0xF0B),
                    0xF43 => array(0xF42, 0xFB7), 0xF4D => array(0xF4C, 0xFB7), 0xF52 => array(0xF51, 0xFB7),
                    0xF57 => array(0xF56, 0xFB7), 0xF5C => array(0xF5B, 0xFB7), 0xF69 => array(0xF40, 0xFB5),
                    0xF73 => array(0xF71, 0xF72), 0xF75 => array(0xF71, 0xF74), 0xF76 => array(0xFB2, 0xF80),
                    0xF77 => array(0xFB2, 0xF71, 0xF80), 0xF78 => array(0xFB3, 0xF80), 0xF79 => array(0xFB3, 0xF71, 0xF80),
                    0xF81 => array(0xF71, 0xF80), 0xF93 => array(0xF92, 0xFB7), 0xF9D => array(0xF9C, 0xFB7),
                    0xFA2 => array(0xFA1, 0xFB7), 0xFA7 => array(0xFA6, 0xFB7), 0xFAC => array(0xFAB, 0xFB7),
                    0xFB9 => array(0xF90, 0xFB5), 0x10C7 => array(0x2D27), 0x10CD => array(0x2D2D),
                    0x10FC => array(0x10DC), 0x1D2C => array(0x61), 0x1D2D => array(0xE6),
                    0x1D2E => array(0x62), 0x1D30 => array(0x64), 0x1D31 => array(0x65),
                    0x1D32 => array(0x1DD), 0x1D33 => array(0x67), 0x1D34 => array(0x68),
                    0x1D35 => array(0x69), 0x1D36 => array(0x6A), 0x1D37 => array(0x6B),
                    0x1D38 => array(0x6C), 0x1D39 => array(0x6D), 0x1D3A => array(0x6E),
                    0x1D3C => array(0x6F), 0x1D3D => array(0x223), 0x1D3E => array(0x70),
                    0x1D3F => array(0x72), 0x1D40 => array(0x74), 0x1D41 => array(0x75),
                    0x1D42 => array(0x77), 0x1D43 => array(0x61), 0x1D44 => array(0x250),
                    0x1D45 => array(0x251), 0x1D46 => array(0x1D02), 0x1D47 => array(0x62),
                    0x1D48 => array(0x64), 0x1D49 => array(0x65), 0x1D4A => array(0x259),
                    0x1D4B => array(0x25B), 0x1D4C => array(0x25C), 0x1D4D => array(0x67),
                    0x1D4F => array(0x6B), 0x1D50 => array(0x6D), 0x1D51 => array(0x14B),
                    0x1D52 => array(0x6F), 0x1D53 => array(0x254), 0x1D54 => array(0x1D16),
                    0x1D55 => array(0x1D17), 0x1D56 => array(0x70), 0x1D57 => array(0x74),
                    0x1D58 => array(0x75), 0x1D59 => array(0x1D1D), 0x1D5A => array(0x26F),
                    0x1D5B => array(0x76), 0x1D5C => array(0x1D25), 0x1D5D => array(0x3B2),
                    0x1D5E => array(0x3B3), 0x1D5F => array(0x3B4), 0x1D60 => array(0x3C6),
                    0x1D61 => array(0x3C7), 0x1D62 => array(0x69), 0x1D63 => array(0x72),
                    0x1D64 => array(0x75), 0x1D65 => array(0x76), 0x1D66 => array(0x3B2),
                    0x1D67 => array(0x3B3), 0x1D68 => array(0x3C1), 0x1D69 => array(0x3C6),
                    0x1D6A => array(0x3C7), 0x1D78 => array(0x43D), 0x1D9B => array(0x252),
                    0x1D9C => array(0x63), 0x1D9D => array(0x255), 0x1D9E => array(0xF0),
                    0x1D9F => array(0x25C), 0x1DA0 => array(0x66), 0x1DA1 => array(0x25F),
                    0x1DA2 => array(0x261), 0x1DA3 => array(0x265), 0x1DA4 => array(0x268),
                    0x1DA5 => array(0x269), 0x1DA6 => array(0x26A), 0x1DA7 => array(0x1D7B),
                    0x1DA8 => array(0x29D), 0x1DA9 => array(0x26D), 0x1DAA => array(0x1D85),
                    0x1DAB => array(0x29F), 0x1DAC => array(0x271), 0x1DAD => array(0x270),
                    0x1DAE => array(0x272), 0x1DAF => array(0x273), 0x1DB0 => array(0x274),
                    0x1DB1 => array(0x275), 0x1DB2 => array(0x278), 0x1DB3 => array(0x282),
                    0x1DB4 => array(0x283), 0x1DB5 => array(0x1AB), 0x1DB6 => array(0x289),
                    0x1DB7 => array(0x28A), 0x1DB8 => array(0x1D1C), 0x1DB9 => array(0x28B),
                    0x1DBA => array(0x28C), 0x1DBB => array(0x7A), 0x1DBC => array(0x290),
                    0x1DBD => array(0x291), 0x1DBE => array(0x292), 0x1DBF => array(0x3B8),
                    0x1E00 => array(0x1E01), 0x1E02 => array(0x1E03), 0x1E04 => array(0x1E05),
                    0x1E06 => array(0x1E07), 0x1E08 => array(0x1E09), 0x1E0A => array(0x1E0B),
                    0x1E0C => array(0x1E0D), 0x1E0E => array(0x1E0F), 0x1E10 => array(0x1E11),
                    0x1E12 => array(0x1E13), 0x1E14 => array(0x1E15), 0x1E16 => array(0x1E17),
                    0x1E18 => array(0x1E19), 0x1E1A => array(0x1E1B), 0x1E1C => array(0x1E1D),
                    0x1E1E => array(0x1E1F), 0x1E20 => array(0x1E21), 0x1E22 => array(0x1E23),
                    0x1E24 => array(0x1E25), 0x1E26 => array(0x1E27), 0x1E28 => array(0x1E29),
                    0x1E2A => array(0x1E2B), 0x1E2C => array(0x1E2D), 0x1E2E => array(0x1E2F),
                    0x1E30 => array(0x1E31), 0x1E32 => array(0x1E33), 0x1E34 => array(0x1E35),
                    0x1E36 => array(0x1E37), 0x1E38 => array(0x1E39), 0x1E3A => array(0x1E3B),
                    0x1E3C => array(0x1E3D), 0x1E3E => array(0x1E3F), 0x1E40 => array(0x1E41),
                    0x1E42 => array(0x1E43), 0x1E44 => array(0x1E45), 0x1E46 => array(0x1E47),
                    0x1E48 => array(0x1E49), 0x1E4A => array(0x1E4B), 0x1E4C => array(0x1E4D),
                    0x1E4E => array(0x1E4F), 0x1E50 => array(0x1E51), 0x1E52 => array(0x1E53),
                    0x1E54 => array(0x1E55), 0x1E56 => array(0x1E57), 0x1E58 => array(0x1E59),
                    0x1E5A => array(0x1E5B), 0x1E5C => array(0x1E5D), 0x1E5E => array(0x1E5F),
                    0x1E60 => array(0x1E61), 0x1E62 => array(0x1E63), 0x1E64 => array(0x1E65),
                    0x1E66 => array(0x1E67), 0x1E68 => array(0x1E69), 0x1E6A => array(0x1E6B),
                    0x1E6C => array(0x1E6D), 0x1E6E => array(0x1E6F), 0x1E70 => array(0x1E71),
                    0x1E72 => array(0x1E73), 0x1E74 => array(0x1E75), 0x1E76 => array(0x1E77),
                    0x1E78 => array(0x1E79), 0x1E7A => array(0x1E7B), 0x1E7C => array(0x1E7D),
                    0x1E7E => array(0x1E7F), 0x1E80 => array(0x1E81), 0x1E82 => array(0x1E83),
                    0x1E84 => array(0x1E85), 0x1E86 => array(0x1E87), 0x1E88 => array(0x1E89),
                    0x1E8A => array(0x1E8B), 0x1E8C => array(0x1E8D), 0x1E8E => array(0x1E8F),
                    0x1E90 => array(0x1E91), 0x1E92 => array(0x1E93), 0x1E94 => array(0x1E95),
                    0x1E9A => array(0x61, 0x2BE), 0x1E9B => array(0x1E61), 0x1E9E => array(0x73, 0x73),
                    0x1EA0 => array(0x1EA1), 0x1EA2 => array(0x1EA3), 0x1EA4 => array(0x1EA5),
                    0x1EA6 => array(0x1EA7), 0x1EA8 => array(0x1EA9), 0x1EAA => array(0x1EAB),
                    0x1EAC => array(0x1EAD), 0x1EAE => array(0x1EAF), 0x1EB0 => array(0x1EB1),
                    0x1EB2 => array(0x1EB3), 0x1EB4 => array(0x1EB5), 0x1EB6 => array(0x1EB7),
                    0x1EB8 => array(0x1EB9), 0x1EBA => array(0x1EBB), 0x1EBC => array(0x1EBD),
                    0x1EBE => array(0x1EBF), 0x1EC0 => array(0x1EC1), 0x1EC2 => array(0x1EC3),
                    0x1EC4 => array(0x1EC5), 0x1EC6 => array(0x1EC7), 0x1EC8 => array(0x1EC9),
                    0x1ECA => array(0x1ECB), 0x1ECC => array(0x1ECD), 0x1ECE => array(0x1ECF),
                    0x1ED0 => array(0x1ED1), 0x1ED2 => array(0x1ED3), 0x1ED4 => array(0x1ED5),
                    0x1ED6 => array(0x1ED7), 0x1ED8 => array(0x1ED9), 0x1EDA => array(0x1EDB),
                    0x1EDC => array(0x1EDD), 0x1EDE => array(0x1EDF), 0x1EE0 => array(0x1EE1),
                    0x1EE2 => array(0x1EE3), 0x1EE4 => array(0x1EE5), 0x1EE6 => array(0x1EE7),
                    0x1EE8 => array(0x1EE9), 0x1EEA => array(0x1EEB), 0x1EEC => array(0x1EED),
                    0x1EEE => array(0x1EEF), 0x1EF0 => array(0x1EF1), 0x1EF2 => array(0x1EF3),
                    0x1EF4 => array(0x1EF5), 0x1EF6 => array(0x1EF7), 0x1EF8 => array(0x1EF9),
                    0x1EFA => array(0x1EFB), 0x1EFC => array(0x1EFD), 0x1EFE => array(0x1EFF),
                    0x1F08 => array(0x1F00), 0x1F09 => array(0x1F01), 0x1F0A => array(0x1F02),
                    0x1F0B => array(0x1F03), 0x1F0C => array(0x1F04), 0x1F0D => array(0x1F05),
                    0x1F0E => array(0x1F06), 0x1F0F => array(0x1F07), 0x1F18 => array(0x1F10),
                    0x1F19 => array(0x1F11), 0x1F1A => array(0x1F12), 0x1F1B => array(0x1F13),
                    0x1F1C => array(0x1F14), 0x1F1D => array(0x1F15), 0x1F28 => array(0x1F20),
                    0x1F29 => array(0x1F21), 0x1F2A => array(0x1F22), 0x1F2B => array(0x1F23),
                    0x1F2C => array(0x1F24), 0x1F2D => array(0x1F25), 0x1F2E => array(0x1F26),
                    0x1F2F => array(0x1F27), 0x1F38 => array(0x1F30), 0x1F39 => array(0x1F31),
                    0x1F3A => array(0x1F32), 0x1F3B => array(0x1F33), 0x1F3C => array(0x1F34),
                    0x1F3D => array(0x1F35), 0x1F3E => array(0x1F36), 0x1F3F => array(0x1F37),
                    0x1F48 => array(0x1F40), 0x1F49 => array(0x1F41), 0x1F4A => array(0x1F42),
                    0x1F4B => array(0x1F43), 0x1F4C => array(0x1F44), 0x1F4D => array(0x1F45),
                    0x1F59 => array(0x1F51), 0x1F5B => array(0x1F53), 0x1F5D => array(0x1F55),
                    0x1F5F => array(0x1F57), 0x1F68 => array(0x1F60), 0x1F69 => array(0x1F61),
                    0x1F6A => array(0x1F62), 0x1F6B => array(0x1F63), 0x1F6C => array(0x1F64),
                    0x1F6D => array(0x1F65), 0x1F6E => array(0x1F66), 0x1F6F => array(0x1F67),
                    0x1F71 => array(0x3AC), 0x1F73 => array(0x3AD), 0x1F75 => array(0x3AE),
                    0x1F77 => array(0x3AF), 0x1F79 => array(0x3CC), 0x1F7B => array(0x3CD),
                    0x1F7D => array(0x3CE), 0x1F80 => array(0x1F00, 0x3B9), 0x1F81 => array(0x1F01, 0x3B9),
                    0x1F82 => array(0x1F02, 0x3B9), 0x1F83 => array(0x1F03, 0x3B9), 0x1F84 => array(0x1F04, 0x3B9),
                    0x1F85 => array(0x1F05, 0x3B9), 0x1F86 => array(0x1F06, 0x3B9), 0x1F87 => array(0x1F07, 0x3B9),
                    0x1F88 => array(0x1F00, 0x3B9), 0x1F89 => array(0x1F01, 0x3B9), 0x1F8A => array(0x1F02, 0x3B9),
                    0x1F8B => array(0x1F03, 0x3B9), 0x1F8C => array(0x1F04, 0x3B9), 0x1F8D => array(0x1F05, 0x3B9),
                    0x1F8E => array(0x1F06, 0x3B9), 0x1F8F => array(0x1F07, 0x3B9), 0x1F90 => array(0x1F20, 0x3B9),
                    0x1F91 => array(0x1F21, 0x3B9), 0x1F92 => array(0x1F22, 0x3B9), 0x1F93 => array(0x1F23, 0x3B9),
                    0x1F94 => array(0x1F24, 0x3B9), 0x1F95 => array(0x1F25, 0x3B9), 0x1F96 => array(0x1F26, 0x3B9),
                    0x1F97 => array(0x1F27, 0x3B9), 0x1F98 => array(0x1F20, 0x3B9), 0x1F99 => array(0x1F21, 0x3B9),
                    0x1F9A => array(0x1F22, 0x3B9), 0x1F9B => array(0x1F23, 0x3B9), 0x1F9C => array(0x1F24, 0x3B9),
                    0x1F9D => array(0x1F25, 0x3B9), 0x1F9E => array(0x1F26, 0x3B9), 0x1F9F => array(0x1F27, 0x3B9),
                    0x1FA0 => array(0x1F60, 0x3B9), 0x1FA1 => array(0x1F61, 0x3B9), 0x1FA2 => array(0x1F62, 0x3B9),
                    0x1FA3 => array(0x1F63, 0x3B9), 0x1FA4 => array(0x1F64, 0x3B9), 0x1FA5 => array(0x1F65, 0x3B9),
                    0x1FA6 => array(0x1F66, 0x3B9), 0x1FA7 => array(0x1F67, 0x3B9), 0x1FA8 => array(0x1F60, 0x3B9),
                    0x1FA9 => array(0x1F61, 0x3B9), 0x1FAA => array(0x1F62, 0x3B9), 0x1FAB => array(0x1F63, 0x3B9),
                    0x1FAC => array(0x1F64, 0x3B9), 0x1FAD => array(0x1F65, 0x3B9), 0x1FAE => array(0x1F66, 0x3B9),
                    0x1FAF => array(0x1F67, 0x3B9), 0x1FB2 => array(0x1F70, 0x3B9), 0x1FB3 => array(0x3B1, 0x3B9),
                    0x1FB4 => array(0x3AC, 0x3B9), 0x1FB7 => array(0x1FB6, 0x3B9), 0x1FB8 => array(0x1FB0),
                    0x1FB9 => array(0x1FB1), 0x1FBA => array(0x1F70), 0x1FBB => array(0x3AC),
                    0x1FBC => array(0x3B1, 0x3B9), 0x1FBE => array(0x3B9), 0x1FC2 => array(0x1F74, 0x3B9),
                    0x1FC3 => array(0x3B7, 0x3B9), 0x1FC4 => array(0x3AE, 0x3B9), 0x1FC7 => array(0x1FC6, 0x3B9),
                    0x1FC8 => array(0x1F72), 0x1FC9 => array(0x3AD), 0x1FCA => array(0x1F74),
                    0x1FCB => array(0x3AE), 0x1FCC => array(0x3B7, 0x3B9), 0x1FD3 => array(0x390),
                    0x1FD8 => array(0x1FD0), 0x1FD9 => array(0x1FD1), 0x1FDA => array(0x1F76),
                    0x1FDB => array(0x3AF), 0x1FE3 => array(0x3B0), 0x1FE8 => array(0x1FE0),
                    0x1FE9 => array(0x1FE1), 0x1FEA => array(0x1F7A), 0x1FEB => array(0x3CD),
                    0x1FEC => array(0x1FE5), 0x1FF2 => array(0x1F7C, 0x3B9), 0x1FF3 => array(0x3C9, 0x3B9),
                    0x1FF4 => array(0x3CE, 0x3B9), 0x1FF7 => array(0x1FF6, 0x3B9), 0x1FF8 => array(0x1F78),
                    0x1FF9 => array(0x3CC), 0x1FFA => array(0x1F7C), 0x1FFB => array(0x3CE),
                    0x1FFC => array(0x3C9, 0x3B9), 0x2011 => array(0x2010), 0x2033 => array(0x2032, 0x2032),
                    0x2034 => array(0x2032, 0x2032, 0x2032), 0x2036 => array(0x2035, 0x2035), 0x2037 => array(0x2035, 0x2035, 0x2035),
                    0x2057 => array(0x2032, 0x2032, 0x2032, 0x2032), 0x2070 => array(0x30), 0x2071 => array(0x69),
                    0x2074 => array(0x34), 0x2075 => array(0x35), 0x2076 => array(0x36),
                    0x2077 => array(0x37), 0x2078 => array(0x38), 0x2079 => array(0x39),
                    0x207B => array(0x2212), 0x207F => array(0x6E), 0x2080 => array(0x30),
                    0x2081 => array(0x31), 0x2082 => array(0x32), 0x2083 => array(0x33),
                    0x2084 => array(0x34), 0x2085 => array(0x35), 0x2086 => array(0x36),
                    0x2087 => array(0x37), 0x2088 => array(0x38), 0x2089 => array(0x39),
                    0x208B => array(0x2212), 0x2090 => array(0x61), 0x2091 => array(0x65),
                    0x2092 => array(0x6F), 0x2093 => array(0x78), 0x2094 => array(0x259),
                    0x2095 => array(0x68), 0x2096 => array(0x6B), 0x2097 => array(0x6C),
                    0x2098 => array(0x6D), 0x2099 => array(0x6E), 0x209A => array(0x70),
                    0x209B => array(0x73), 0x209C => array(0x74), 0x20A8 => array(0x72, 0x73),
                    0x2102 => array(0x63), 0x2103 => array(0xB0, 0x63), 0x2107 => array(0x25B),
                    0x2109 => array(0xB0, 0x66), 0x210A => array(0x67), 0x210B => array(0x68),
                    0x210C => array(0x68), 0x210D => array(0x68), 0x210E => array(0x68),
                    0x210F => array(0x127), 0x2110 => array(0x69), 0x2111 => array(0x69),
                    0x2112 => array(0x6C), 0x2113 => array(0x6C), 0x2115 => array(0x6E),
                    0x2116 => array(0x6E, 0x6F), 0x2119 => array(0x70), 0x211A => array(0x71),
                    0x211B => array(0x72), 0x211C => array(0x72), 0x211D => array(0x72),
                    0x2120 => array(0x73, 0x6D), 0x2121 => array(0x74, 0x65, 0x6C), 0x2122 => array(0x74, 0x6D),
                    0x2124 => array(0x7A), 0x2126 => array(0x3C9), 0x2128 => array(0x7A),
                    0x212A => array(0x6B), 0x212B => array(0xE5), 0x212C => array(0x62),
                    0x212D => array(0x63), 0x212F => array(0x65), 0x2130 => array(0x65),
                    0x2131 => array(0x66), 0x2133 => array(0x6D), 0x2134 => array(0x6F),
                    0x2135 => array(0x5D0), 0x2136 => array(0x5D1), 0x2137 => array(0x5D2),
                    0x2138 => array(0x5D3), 0x2139 => array(0x69), 0x213B => array(0x66, 0x61, 0x78),
                    0x213C => array(0x3C0), 0x213D => array(0x3B3), 0x213E => array(0x3B3),
                    0x213F => array(0x3C0), 0x2140 => array(0x2211), 0x2145 => array(0x64),
                    0x2146 => array(0x64), 0x2147 => array(0x65), 0x2148 => array(0x69),
                    0x2149 => array(0x6A), 0x2150 => array(0x31, 0x2044, 0x37), 0x2151 => array(0x31, 0x2044, 0x39),
                    0x2152 => array(0x31, 0x2044, 0x31, 0x30), 0x2153 => array(0x31, 0x2044, 0x33), 0x2154 => array(0x32, 0x2044, 0x33),
                    0x2155 => array(0x31, 0x2044, 0x35), 0x2156 => array(0x32, 0x2044, 0x35), 0x2157 => array(0x33, 0x2044, 0x35),
                    0x2158 => array(0x34, 0x2044, 0x35), 0x2159 => array(0x31, 0x2044, 0x36), 0x215A => array(0x35, 0x2044, 0x36),
                    0x215B => array(0x31, 0x2044, 0x38), 0x215C => array(0x33, 0x2044, 0x38), 0x215D => array(0x35, 0x2044, 0x38),
                    0x215E => array(0x37, 0x2044, 0x38), 0x215F => array(0x31, 0x2044), 0x2160 => array(0x69),
                    0x2161 => array(0x69, 0x69), 0x2162 => array(0x69, 0x69, 0x69), 0x2163 => array(0x69, 0x76),
                    0x2164 => array(0x76), 0x2165 => array(0x76, 0x69), 0x2166 => array(0x76, 0x69, 0x69),
                    0x2167 => array(0x76, 0x69, 0x69, 0x69), 0x2168 => array(0x69, 0x78), 0x2169 => array(0x78),
                    0x216A => array(0x78, 0x69), 0x216B => array(0x78, 0x69, 0x69), 0x216C => array(0x6C),
                    0x216D => array(0x63), 0x216E => array(0x64), 0x216F => array(0x6D),
                    0x2170 => array(0x69), 0x2171 => array(0x69, 0x69), 0x2172 => array(0x69, 0x69, 0x69),
                    0x2173 => array(0x69, 0x76), 0x2174 => array(0x76), 0x2175 => array(0x76, 0x69),
                    0x2176 => array(0x76, 0x69, 0x69), 0x2177 => array(0x76, 0x69, 0x69, 0x69), 0x2178 => array(0x69, 0x78),
                    0x2179 => array(0x78), 0x217A => array(0x78, 0x69), 0x217B => array(0x78, 0x69, 0x69),
                    0x217C => array(0x6C), 0x217D => array(0x63), 0x217E => array(0x64),
                    0x217F => array(0x6D), 0x2189 => array(0x30, 0x2044, 0x33), 0x222C => array(0x222B, 0x222B),
                    0x222D => array(0x222B, 0x222B, 0x222B), 0x222F => array(0x222E, 0x222E), 0x2230 => array(0x222E, 0x222E, 0x222E),
                    0x2329 => array(0x3008), 0x232A => array(0x3009), 0x2460 => array(0x31),
                    0x2461 => array(0x32), 0x2462 => array(0x33), 0x2463 => array(0x34),
                    0x2464 => array(0x35), 0x2465 => array(0x36), 0x2466 => array(0x37),
                    0x2467 => array(0x38), 0x2468 => array(0x39), 0x2469 => array(0x31, 0x30),
                    0x246A => array(0x31, 0x31), 0x246B => array(0x31, 0x32), 0x246C => array(0x31, 0x33),
                    0x246D => array(0x31, 0x34), 0x246E => array(0x31, 0x35), 0x246F => array(0x31, 0x36),
                    0x2470 => array(0x31, 0x37), 0x2471 => array(0x31, 0x38), 0x2472 => array(0x31, 0x39),
                    0x2473 => array(0x32, 0x30), 0x24B6 => array(0x61), 0x24B7 => array(0x62),
                    0x24B8 => array(0x63), 0x24B9 => array(0x64), 0x24BA => array(0x65),
                    0x24BB => array(0x66), 0x24BC => array(0x67), 0x24BD => array(0x68),
                    0x24BE => array(0x69), 0x24BF => array(0x6A), 0x24C0 => array(0x6B),
                    0x24C1 => array(0x6C), 0x24C2 => array(0x6D), 0x24C3 => array(0x6E),
                    0x24C4 => array(0x6F), 0x24C5 => array(0x70), 0x24C6 => array(0x71),
                    0x24C7 => array(0x72), 0x24C8 => array(0x73), 0x24C9 => array(0x74),
                    0x24CA => array(0x75), 0x24CB => array(0x76), 0x24CC => array(0x77),
                    0x24CD => array(0x78), 0x24CE => array(0x79), 0x24CF => array(0x7A),
                    0x24D0 => array(0x61), 0x24D1 => array(0x62), 0x24D2 => array(0x63),
                    0x24D3 => array(0x64), 0x24D4 => array(0x65), 0x24D5 => array(0x66),
                    0x24D6 => array(0x67), 0x24D7 => array(0x68), 0x24D8 => array(0x69),
                    0x24D9 => array(0x6A), 0x24DA => array(0x6B), 0x24DB => array(0x6C),
                    0x24DC => array(0x6D), 0x24DD => array(0x6E), 0x24DE => array(0x6F),
                    0x24DF => array(0x70), 0x24E0 => array(0x71), 0x24E1 => array(0x72),
                    0x24E2 => array(0x73), 0x24E3 => array(0x74), 0x24E4 => array(0x75),
                    0x24E5 => array(0x76), 0x24E6 => array(0x77), 0x24E7 => array(0x78),
                    0x24E8 => array(0x79), 0x24E9 => array(0x7A), 0x24EA => array(0x30),
                    0x2A0C => array(0x222B, 0x222B, 0x222B, 0x222B), 0x2ADC => array(0x2ADD, 0x338), 0x2C00 => array(0x2C30),
                    0x2C01 => array(0x2C31), 0x2C02 => array(0x2C32), 0x2C03 => array(0x2C33),
                    0x2C04 => array(0x2C34), 0x2C05 => array(0x2C35), 0x2C06 => array(0x2C36),
                    0x2C07 => array(0x2C37), 0x2C08 => array(0x2C38), 0x2C09 => array(0x2C39),
                    0x2C0A => array(0x2C3A), 0x2C0B => array(0x2C3B), 0x2C0C => array(0x2C3C),
                    0x2C0D => array(0x2C3D), 0x2C0E => array(0x2C3E), 0x2C0F => array(0x2C3F),
                    0x2C10 => array(0x2C40), 0x2C11 => array(0x2C41), 0x2C12 => array(0x2C42),
                    0x2C13 => array(0x2C43), 0x2C14 => array(0x2C44), 0x2C15 => array(0x2C45),
                    0x2C16 => array(0x2C46), 0x2C17 => array(0x2C47), 0x2C18 => array(0x2C48),
                    0x2C19 => array(0x2C49), 0x2C1A => array(0x2C4A), 0x2C1B => array(0x2C4B),
                    0x2C1C => array(0x2C4C), 0x2C1D => array(0x2C4D), 0x2C1E => array(0x2C4E),
                    0x2C1F => array(0x2C4F), 0x2C20 => array(0x2C50), 0x2C21 => array(0x2C51),
                    0x2C22 => array(0x2C52), 0x2C23 => array(0x2C53), 0x2C24 => array(0x2C54),
                    0x2C25 => array(0x2C55), 0x2C26 => array(0x2C56), 0x2C27 => array(0x2C57),
                    0x2C28 => array(0x2C58), 0x2C29 => array(0x2C59), 0x2C2A => array(0x2C5A),
                    0x2C2B => array(0x2C5B), 0x2C2C => array(0x2C5C), 0x2C2D => array(0x2C5D),
                    0x2C2E => array(0x2C5E), 0x2C60 => array(0x2C61), 0x2C62 => array(0x26B),
                    0x2C63 => array(0x1D7D), 0x2C64 => array(0x27D), 0x2C67 => array(0x2C68),
                    0x2C69 => array(0x2C6A), 0x2C6B => array(0x2C6C), 0x2C6D => array(0x251),
                    0x2C6E => array(0x271), 0x2C6F => array(0x250), 0x2C70 => array(0x252),
                    0x2C72 => array(0x2C73), 0x2C75 => array(0x2C76), 0x2C7C => array(0x6A),
                    0x2C7D => array(0x76), 0x2C7E => array(0x23F), 0x2C7F => array(0x240),
                    0x2C80 => array(0x2C81), 0x2C82 => array(0x2C83), 0x2C84 => array(0x2C85),
                    0x2C86 => array(0x2C87), 0x2C88 => array(0x2C89), 0x2C8A => array(0x2C8B),
                    0x2C8C => array(0x2C8D), 0x2C8E => array(0x2C8F), 0x2C90 => array(0x2C91),
                    0x2C92 => array(0x2C93), 0x2C94 => array(0x2C95), 0x2C96 => array(0x2C97),
                    0x2C98 => array(0x2C99), 0x2C9A => array(0x2C9B), 0x2C9C => array(0x2C9D),
                    0x2C9E => array(0x2C9F), 0x2CA0 => array(0x2CA1), 0x2CA2 => array(0x2CA3),
                    0x2CA4 => array(0x2CA5), 0x2CA6 => array(0x2CA7), 0x2CA8 => array(0x2CA9),
                    0x2CAA => array(0x2CAB), 0x2CAC => array(0x2CAD), 0x2CAE => array(0x2CAF),
                    0x2CB0 => array(0x2CB1), 0x2CB2 => array(0x2CB3), 0x2CB4 => array(0x2CB5),
                    0x2CB6 => array(0x2CB7), 0x2CB8 => array(0x2CB9), 0x2CBA => array(0x2CBB),
                    0x2CBC => array(0x2CBD), 0x2CBE => array(0x2CBF), 0x2CC0 => array(0x2CC1),
                    0x2CC2 => array(0x2CC3), 0x2CC4 => array(0x2CC5), 0x2CC6 => array(0x2CC7),
                    0x2CC8 => array(0x2CC9), 0x2CCA => array(0x2CCB), 0x2CCC => array(0x2CCD),
                    0x2CCE => array(0x2CCF), 0x2CD0 => array(0x2CD1), 0x2CD2 => array(0x2CD3),
                    0x2CD4 => array(0x2CD5), 0x2CD6 => array(0x2CD7), 0x2CD8 => array(0x2CD9),
                    0x2CDA => array(0x2CDB), 0x2CDC => array(0x2CDD), 0x2CDE => array(0x2CDF),
                    0x2CE0 => array(0x2CE1), 0x2CE2 => array(0x2CE3), 0x2CEB => array(0x2CEC),
                    0x2CED => array(0x2CEE), 0x2CF2 => array(0x2CF3), 0x2D6F => array(0x2D61),
                    0x2E9F => array(0x6BCD), 0x2EF3 => array(0x9F9F), 0x2F00 => array(0x4E00),
                    0x2F01 => array(0x4E28), 0x2F02 => array(0x4E36), 0x2F03 => array(0x4E3F),
                    0x2F04 => array(0x4E59), 0x2F05 => array(0x4E85), 0x2F06 => array(0x4E8C),
                    0x2F07 => array(0x4EA0), 0x2F08 => array(0x4EBA), 0x2F09 => array(0x513F),
                    0x2F0A => array(0x5165), 0x2F0B => array(0x516B), 0x2F0C => array(0x5182),
                    0x2F0D => array(0x5196), 0x2F0E => array(0x51AB), 0x2F0F => array(0x51E0),
                    0x2F10 => array(0x51F5), 0x2F11 => array(0x5200), 0x2F12 => array(0x529B),
                    0x2F13 => array(0x52F9), 0x2F14 => array(0x5315), 0x2F15 => array(0x531A),
                    0x2F16 => array(0x5338), 0x2F17 => array(0x5341), 0x2F18 => array(0x535C),
                    0x2F19 => array(0x5369), 0x2F1A => array(0x5382), 0x2F1B => array(0x53B6),
                    0x2F1C => array(0x53C8), 0x2F1D => array(0x53E3), 0x2F1E => array(0x56D7),
                    0x2F1F => array(0x571F), 0x2F20 => array(0x58EB), 0x2F21 => array(0x5902),
                    0x2F22 => array(0x590A), 0x2F23 => array(0x5915), 0x2F24 => array(0x5927),
                    0x2F25 => array(0x5973), 0x2F26 => array(0x5B50), 0x2F27 => array(0x5B80),
                    0x2F28 => array(0x5BF8), 0x2F29 => array(0x5C0F), 0x2F2A => array(0x5C22),
                    0x2F2B => array(0x5C38), 0x2F2C => array(0x5C6E), 0x2F2D => array(0x5C71),
                    0x2F2E => array(0x5DDB), 0x2F2F => array(0x5DE5), 0x2F30 => array(0x5DF1),
                    0x2F31 => array(0x5DFE), 0x2F32 => array(0x5E72), 0x2F33 => array(0x5E7A),
                    0x2F34 => array(0x5E7F), 0x2F35 => array(0x5EF4), 0x2F36 => array(0x5EFE),
                    0x2F37 => array(0x5F0B), 0x2F38 => array(0x5F13), 0x2F39 => array(0x5F50),
                    0x2F3A => array(0x5F61), 0x2F3B => array(0x5F73), 0x2F3C => array(0x5FC3),
                    0x2F3D => array(0x6208), 0x2F3E => array(0x6236), 0x2F3F => array(0x624B),
                    0x2F40 => array(0x652F), 0x2F41 => array(0x6534), 0x2F42 => array(0x6587),
                    0x2F43 => array(0x6597), 0x2F44 => array(0x65A4), 0x2F45 => array(0x65B9),
                    0x2F46 => array(0x65E0), 0x2F47 => array(0x65E5), 0x2F48 => array(0x66F0),
                    0x2F49 => array(0x6708), 0x2F4A => array(0x6728), 0x2F4B => array(0x6B20),
                    0x2F4C => array(0x6B62), 0x2F4D => array(0x6B79), 0x2F4E => array(0x6BB3),
                    0x2F4F => array(0x6BCB), 0x2F50 => array(0x6BD4), 0x2F51 => array(0x6BDB),
                    0x2F52 => array(0x6C0F), 0x2F53 => array(0x6C14), 0x2F54 => array(0x6C34),
                    0x2F55 => array(0x706B), 0x2F56 => array(0x722A), 0x2F57 => array(0x7236),
                    0x2F58 => array(0x723B), 0x2F59 => array(0x723F), 0x2F5A => array(0x7247),
                    0x2F5B => array(0x7259), 0x2F5C => array(0x725B), 0x2F5D => array(0x72AC),
                    0x2F5E => array(0x7384), 0x2F5F => array(0x7389), 0x2F60 => array(0x74DC),
                    0x2F61 => array(0x74E6), 0x2F62 => array(0x7518), 0x2F63 => array(0x751F),
                    0x2F64 => array(0x7528), 0x2F65 => array(0x7530), 0x2F66 => array(0x758B),
                    0x2F67 => array(0x7592), 0x2F68 => array(0x7676), 0x2F69 => array(0x767D),
                    0x2F6A => array(0x76AE), 0x2F6B => array(0x76BF), 0x2F6C => array(0x76EE),
                    0x2F6D => array(0x77DB), 0x2F6E => array(0x77E2), 0x2F6F => array(0x77F3),
                    0x2F70 => array(0x793A), 0x2F71 => array(0x79B8), 0x2F72 => array(0x79BE),
                    0x2F73 => array(0x7A74), 0x2F74 => array(0x7ACB), 0x2F75 => array(0x7AF9),
                    0x2F76 => array(0x7C73), 0x2F77 => array(0x7CF8), 0x2F78 => array(0x7F36),
                    0x2F79 => array(0x7F51), 0x2F7A => array(0x7F8A), 0x2F7B => array(0x7FBD),
                    0x2F7C => array(0x8001), 0x2F7D => array(0x800C), 0x2F7E => array(0x8012),
                    0x2F7F => array(0x8033), 0x2F80 => array(0x807F), 0x2F81 => array(0x8089),
                    0x2F82 => array(0x81E3), 0x2F83 => array(0x81EA), 0x2F84 => array(0x81F3),
                    0x2F85 => array(0x81FC), 0x2F86 => array(0x820C), 0x2F87 => array(0x821B),
                    0x2F88 => array(0x821F), 0x2F89 => array(0x826E), 0x2F8A => array(0x8272),
                    0x2F8B => array(0x8278), 0x2F8C => array(0x864D), 0x2F8D => array(0x866B),
                    0x2F8E => array(0x8840), 0x2F8F => array(0x884C), 0x2F90 => array(0x8863),
                    0x2F91 => array(0x897E), 0x2F92 => array(0x898B), 0x2F93 => array(0x89D2),
                    0x2F94 => array(0x8A00), 0x2F95 => array(0x8C37), 0x2F96 => array(0x8C46),
                    0x2F97 => array(0x8C55), 0x2F98 => array(0x8C78), 0x2F99 => array(0x8C9D),
                    0x2F9A => array(0x8D64), 0x2F9B => array(0x8D70), 0x2F9C => array(0x8DB3),
                    0x2F9D => array(0x8EAB), 0x2F9E => array(0x8ECA), 0x2F9F => array(0x8F9B),
                    0x2FA0 => array(0x8FB0), 0x2FA1 => array(0x8FB5), 0x2FA2 => array(0x9091),
                    0x2FA3 => array(0x9149), 0x2FA4 => array(0x91C6), 0x2FA5 => array(0x91CC),
                    0x2FA6 => array(0x91D1), 0x2FA7 => array(0x9577), 0x2FA8 => array(0x9580),
                    0x2FA9 => array(0x961C), 0x2FAA => array(0x96B6), 0x2FAB => array(0x96B9),
                    0x2FAC => array(0x96E8), 0x2FAD => array(0x9751), 0x2FAE => array(0x975E),
                    0x2FAF => array(0x9762), 0x2FB0 => array(0x9769), 0x2FB1 => array(0x97CB),
                    0x2FB2 => array(0x97ED), 0x2FB3 => array(0x97F3), 0x2FB4 => array(0x9801),
                    0x2FB5 => array(0x98A8), 0x2FB6 => array(0x98DB), 0x2FB7 => array(0x98DF),
                    0x2FB8 => array(0x9996), 0x2FB9 => array(0x9999), 0x2FBA => array(0x99AC),
                    0x2FBB => array(0x9AA8), 0x2FBC => array(0x9AD8), 0x2FBD => array(0x9ADF),
                    0x2FBE => array(0x9B25), 0x2FBF => array(0x9B2F), 0x2FC0 => array(0x9B32),
                    0x2FC1 => array(0x9B3C), 0x2FC2 => array(0x9B5A), 0x2FC3 => array(0x9CE5),
                    0x2FC4 => array(0x9E75), 0x2FC5 => array(0x9E7F), 0x2FC6 => array(0x9EA5),
                    0x2FC7 => array(0x9EBB), 0x2FC8 => array(0x9EC3), 0x2FC9 => array(0x9ECD),
                    0x2FCA => array(0x9ED1), 0x2FCB => array(0x9EF9), 0x2FCC => array(0x9EFD),
                    0x2FCD => array(0x9F0E), 0x2FCE => array(0x9F13), 0x2FCF => array(0x9F20),
                    0x2FD0 => array(0x9F3B), 0x2FD1 => array(0x9F4A), 0x2FD2 => array(0x9F52),
                    0x2FD3 => array(0x9F8D), 0x2FD4 => array(0x9F9C), 0x2FD5 => array(0x9FA0),
                    0x3002 => array(0x2E), 0x3036 => array(0x3012), 0x3038 => array(0x5341),
                    0x3039 => array(0x5344), 0x303A => array(0x5345), 0x309F => array(0x3088, 0x308A),
                    0x30FF => array(0x30B3, 0x30C8), 0x3131 => array(0x1100), 0x3132 => array(0x1101),
                    0x3133 => array(0x11AA), 0x3134 => array(0x1102), 0x3135 => array(0x11AC),
                    0x3136 => array(0x11AD), 0x3137 => array(0x1103), 0x3138 => array(0x1104),
                    0x3139 => array(0x1105), 0x313A => array(0x11B0), 0x313B => array(0x11B1),
                    0x313C => array(0x11B2), 0x313D => array(0x11B3), 0x313E => array(0x11B4),
                    0x313F => array(0x11B5), 0x3140 => array(0x111A), 0x3141 => array(0x1106),
                    0x3142 => array(0x1107), 0x3143 => array(0x1108), 0x3144 => array(0x1121),
                    0x3145 => array(0x1109), 0x3146 => array(0x110A), 0x3147 => array(0x110B),
                    0x3148 => array(0x110C), 0x3149 => array(0x110D), 0x314A => array(0x110E),
                    0x314B => array(0x110F), 0x314C => array(0x1110), 0x314D => array(0x1111),
                    0x314E => array(0x1112), 0x314F => array(0x1161), 0x3150 => array(0x1162),
                    0x3151 => array(0x1163), 0x3152 => array(0x1164), 0x3153 => array(0x1165),
                    0x3154 => array(0x1166), 0x3155 => array(0x1167), 0x3156 => array(0x1168),
                    0x3157 => array(0x1169), 0x3158 => array(0x116A), 0x3159 => array(0x116B),
                    0x315A => array(0x116C), 0x315B => array(0x116D), 0x315C => array(0x116E),
                    0x315D => array(0x116F), 0x315E => array(0x1170), 0x315F => array(0x1171),
                    0x3160 => array(0x1172), 0x3161 => array(0x1173), 0x3162 => array(0x1174),
                    0x3163 => array(0x1175), 0x3165 => array(0x1114), 0x3166 => array(0x1115),
                    0x3167 => array(0x11C7), 0x3168 => array(0x11C8), 0x3169 => array(0x11CC),
                    0x316A => array(0x11CE), 0x316B => array(0x11D3), 0x316C => array(0x11D7),
                    0x316D => array(0x11D9), 0x316E => array(0x111C), 0x316F => array(0x11DD),
                    0x3170 => array(0x11DF), 0x3171 => array(0x111D), 0x3172 => array(0x111E),
                    0x3173 => array(0x1120), 0x3174 => array(0x1122), 0x3175 => array(0x1123),
                    0x3176 => array(0x1127), 0x3177 => array(0x1129), 0x3178 => array(0x112B),
                    0x3179 => array(0x112C), 0x317A => array(0x112D), 0x317B => array(0x112E),
                    0x317C => array(0x112F), 0x317D => array(0x1132), 0x317E => array(0x1136),
                    0x317F => array(0x1140), 0x3180 => array(0x1147), 0x3181 => array(0x114C),
                    0x3182 => array(0x11F1), 0x3183 => array(0x11F2), 0x3184 => array(0x1157),
                    0x3185 => array(0x1158), 0x3186 => array(0x1159), 0x3187 => array(0x1184),
                    0x3188 => array(0x1185), 0x3189 => array(0x1188), 0x318A => array(0x1191),
                    0x318B => array(0x1192), 0x318C => array(0x1194), 0x318D => array(0x119E),
                    0x318E => array(0x11A1), 0x3192 => array(0x4E00), 0x3193 => array(0x4E8C),
                    0x3194 => array(0x4E09), 0x3195 => array(0x56DB), 0x3196 => array(0x4E0A),
                    0x3197 => array(0x4E2D), 0x3198 => array(0x4E0B), 0x3199 => array(0x7532),
                    0x319A => array(0x4E59), 0x319B => array(0x4E19), 0x319C => array(0x4E01),
                    0x319D => array(0x5929), 0x319E => array(0x5730), 0x319F => array(0x4EBA),
                    0x3244 => array(0x554F), 0x3245 => array(0x5E7C), 0x3246 => array(0x6587),
                    0x3247 => array(0x7B8F), 0x3250 => array(0x70, 0x74, 0x65), 0x3251 => array(0x32, 0x31),
                    0x3252 => array(0x32, 0x32), 0x3253 => array(0x32, 0x33), 0x3254 => array(0x32, 0x34),
                    0x3255 => array(0x32, 0x35), 0x3256 => array(0x32, 0x36), 0x3257 => array(0x32, 0x37),
                    0x3258 => array(0x32, 0x38), 0x3259 => array(0x32, 0x39), 0x325A => array(0x33, 0x30),
                    0x325B => array(0x33, 0x31), 0x325C => array(0x33, 0x32), 0x325D => array(0x33, 0x33),
                    0x325E => array(0x33, 0x34), 0x325F => array(0x33, 0x35), 0x3260 => array(0x1100),
                    0x3261 => array(0x1102), 0x3262 => array(0x1103), 0x3263 => array(0x1105),
                    0x3264 => array(0x1106), 0x3265 => array(0x1107), 0x3266 => array(0x1109),
                    0x3267 => array(0x110B), 0x3268 => array(0x110C), 0x3269 => array(0x110E),
                    0x326A => array(0x110F), 0x326B => array(0x1110), 0x326C => array(0x1111),
                    0x326D => array(0x1112), 0x326E => array(0xAC00), 0x326F => array(0xB098),
                    0x3270 => array(0xB2E4), 0x3271 => array(0xB77C), 0x3272 => array(0xB9C8),
                    0x3273 => array(0xBC14), 0x3274 => array(0xC0AC), 0x3275 => array(0xC544),
                    0x3276 => array(0xC790), 0x3277 => array(0xCC28), 0x3278 => array(0xCE74),
                    0x3279 => array(0xD0C0), 0x327A => array(0xD30C), 0x327B => array(0xD558),
                    0x327C => array(0xCC38, 0xACE0), 0x327D => array(0xC8FC, 0xC758), 0x327E => array(0xC6B0),
                    0x3280 => array(0x4E00), 0x3281 => array(0x4E8C), 0x3282 => array(0x4E09),
                    0x3283 => array(0x56DB), 0x3284 => array(0x4E94), 0x3285 => array(0x516D),
                    0x3286 => array(0x4E03), 0x3287 => array(0x516B), 0x3288 => array(0x4E5D),
                    0x3289 => array(0x5341), 0x328A => array(0x6708), 0x328B => array(0x706B),
                    0x328C => array(0x6C34), 0x328D => array(0x6728), 0x328E => array(0x91D1),
                    0x328F => array(0x571F), 0x3290 => array(0x65E5), 0x3291 => array(0x682A),
                    0x3292 => array(0x6709), 0x3293 => array(0x793E), 0x3294 => array(0x540D),
                    0x3295 => array(0x7279), 0x3296 => array(0x8CA1), 0x3297 => array(0x795D),
                    0x3298 => array(0x52B4), 0x3299 => array(0x79D8), 0x329A => array(0x7537),
                    0x329B => array(0x5973), 0x329C => array(0x9069), 0x329D => array(0x512A),
                    0x329E => array(0x5370), 0x329F => array(0x6CE8), 0x32A0 => array(0x9805),
                    0x32A1 => array(0x4F11), 0x32A2 => array(0x5199), 0x32A3 => array(0x6B63),
                    0x32A4 => array(0x4E0A), 0x32A5 => array(0x4E2D), 0x32A6 => array(0x4E0B),
                    0x32A7 => array(0x5DE6), 0x32A8 => array(0x53F3), 0x32A9 => array(0x533B),
                    0x32AA => array(0x5B97), 0x32AB => array(0x5B66), 0x32AC => array(0x76E3),
                    0x32AD => array(0x4F01), 0x32AE => array(0x8CC7), 0x32AF => array(0x5354),
                    0x32B0 => array(0x591C), 0x32B1 => array(0x33, 0x36), 0x32B2 => array(0x33, 0x37),
                    0x32B3 => array(0x33, 0x38), 0x32B4 => array(0x33, 0x39), 0x32B5 => array(0x34, 0x30),
                    0x32B6 => array(0x34, 0x31), 0x32B7 => array(0x34, 0x32), 0x32B8 => array(0x34, 0x33),
                    0x32B9 => array(0x34, 0x34), 0x32BA => array(0x34, 0x35), 0x32BB => array(0x34, 0x36),
                    0x32BC => array(0x34, 0x37), 0x32BD => array(0x34, 0x38), 0x32BE => array(0x34, 0x39),
                    0x32BF => array(0x35, 0x30), 0x32C0 => array(0x31, 0x6708), 0x32C1 => array(0x32, 0x6708),
                    0x32C2 => array(0x33, 0x6708), 0x32C3 => array(0x34, 0x6708), 0x32C4 => array(0x35, 0x6708),
                    0x32C5 => array(0x36, 0x6708), 0x32C6 => array(0x37, 0x6708), 0x32C7 => array(0x38, 0x6708),
                    0x32C8 => array(0x39, 0x6708), 0x32C9 => array(0x31, 0x30, 0x6708), 0x32CA => array(0x31, 0x31, 0x6708),
                    0x32CB => array(0x31, 0x32, 0x6708), 0x32CC => array(0x68, 0x67), 0x32CD => array(0x65, 0x72, 0x67),
                    0x32CE => array(0x65, 0x76), 0x32CF => array(0x6C, 0x74, 0x64), 0x32D0 => array(0x30A2),
                    0x32D1 => array(0x30A4), 0x32D2 => array(0x30A6), 0x32D3 => array(0x30A8),
                    0x32D4 => array(0x30AA), 0x32D5 => array(0x30AB), 0x32D6 => array(0x30AD),
                    0x32D7 => array(0x30AF), 0x32D8 => array(0x30B1), 0x32D9 => array(0x30B3),
                    0x32DA => array(0x30B5), 0x32DB => array(0x30B7), 0x32DC => array(0x30B9),
                    0x32DD => array(0x30BB), 0x32DE => array(0x30BD), 0x32DF => array(0x30BF),
                    0x32E0 => array(0x30C1), 0x32E1 => array(0x30C4), 0x32E2 => array(0x30C6),
                    0x32E3 => array(0x30C8), 0x32E4 => array(0x30CA), 0x32E5 => array(0x30CB),
                    0x32E6 => array(0x30CC), 0x32E7 => array(0x30CD), 0x32E8 => array(0x30CE),
                    0x32E9 => array(0x30CF), 0x32EA => array(0x30D2), 0x32EB => array(0x30D5),
                    0x32EC => array(0x30D8), 0x32ED => array(0x30DB), 0x32EE => array(0x30DE),
                    0x32EF => array(0x30DF), 0x32F0 => array(0x30E0), 0x32F1 => array(0x30E1),
                    0x32F2 => array(0x30E2), 0x32F3 => array(0x30E4), 0x32F4 => array(0x30E6),
                    0x32F5 => array(0x30E8), 0x32F6 => array(0x30E9), 0x32F7 => array(0x30EA),
                    0x32F8 => array(0x30EB), 0x32F9 => array(0x30EC), 0x32FA => array(0x30ED),
                    0x32FB => array(0x30EF), 0x32FC => array(0x30F0), 0x32FD => array(0x30F1),
                    0x32FE => array(0x30F2), 0x3300 => array(0x30A2, 0x30D1, 0x30FC, 0x30C8), 0x3301 => array(0x30A2, 0x30EB, 0x30D5, 0x30A1),
                    0x3302 => array(0x30A2, 0x30F3, 0x30DA, 0x30A2), 0x3303 => array(0x30A2, 0x30FC, 0x30EB), 0x3304 => array(0x30A4, 0x30CB, 0x30F3, 0x30B0),
                    0x3305 => array(0x30A4, 0x30F3, 0x30C1), 0x3306 => array(0x30A6, 0x30A9, 0x30F3), 0x3307 => array(0x30A8, 0x30B9, 0x30AF, 0x30FC, 0x30C9),
                    0x3308 => array(0x30A8, 0x30FC, 0x30AB, 0x30FC), 0x3309 => array(0x30AA, 0x30F3, 0x30B9), 0x330A => array(0x30AA, 0x30FC, 0x30E0),
                    0x330B => array(0x30AB, 0x30A4, 0x30EA), 0x330C => array(0x30AB, 0x30E9, 0x30C3, 0x30C8), 0x330D => array(0x30AB, 0x30ED, 0x30EA, 0x30FC),
                    0x330E => array(0x30AC, 0x30ED, 0x30F3), 0x330F => array(0x30AC, 0x30F3, 0x30DE), 0x3310 => array(0x30AE, 0x30AC),
                    0x3311 => array(0x30AE, 0x30CB, 0x30FC), 0x3312 => array(0x30AD, 0x30E5, 0x30EA, 0x30FC), 0x3313 => array(0x30AE, 0x30EB, 0x30C0, 0x30FC),
                    0x3314 => array(0x30AD, 0x30ED), 0x3315 => array(0x30AD, 0x30ED, 0x30B0, 0x30E9, 0x30E0), 0x3316 => array(0x30AD, 0x30ED, 0x30E1, 0x30FC, 0x30C8, 0x30EB),
                    0x3317 => array(0x30AD, 0x30ED, 0x30EF, 0x30C3, 0x30C8), 0x3318 => array(0x30B0, 0x30E9, 0x30E0), 0x3319 => array(0x30B0, 0x30E9, 0x30E0, 0x30C8, 0x30F3),
                    0x331A => array(0x30AF, 0x30EB, 0x30BC, 0x30A4, 0x30ED), 0x331B => array(0x30AF, 0x30ED, 0x30FC, 0x30CD), 0x331C => array(0x30B1, 0x30FC, 0x30B9),
                    0x331D => array(0x30B3, 0x30EB, 0x30CA), 0x331E => array(0x30B3, 0x30FC, 0x30DD), 0x331F => array(0x30B5, 0x30A4, 0x30AF, 0x30EB),
                    0x3320 => array(0x30B5, 0x30F3, 0x30C1, 0x30FC, 0x30E0), 0x3321 => array(0x30B7, 0x30EA, 0x30F3, 0x30B0), 0x3322 => array(0x30BB, 0x30F3, 0x30C1),
                    0x3323 => array(0x30BB, 0x30F3, 0x30C8), 0x3324 => array(0x30C0, 0x30FC, 0x30B9), 0x3325 => array(0x30C7, 0x30B7),
                    0x3326 => array(0x30C9, 0x30EB), 0x3327 => array(0x30C8, 0x30F3), 0x3328 => array(0x30CA, 0x30CE),
                    0x3329 => array(0x30CE, 0x30C3, 0x30C8), 0x332A => array(0x30CF, 0x30A4, 0x30C4), 0x332B => array(0x30D1, 0x30FC, 0x30BB, 0x30F3, 0x30C8),
                    0x332C => array(0x30D1, 0x30FC, 0x30C4), 0x332D => array(0x30D0, 0x30FC, 0x30EC, 0x30EB), 0x332E => array(0x30D4, 0x30A2, 0x30B9, 0x30C8, 0x30EB),
                    0x332F => array(0x30D4, 0x30AF, 0x30EB), 0x3330 => array(0x30D4, 0x30B3), 0x3331 => array(0x30D3, 0x30EB),
                    0x3332 => array(0x30D5, 0x30A1, 0x30E9, 0x30C3, 0x30C9), 0x3333 => array(0x30D5, 0x30A3, 0x30FC, 0x30C8), 0x3334 => array(0x30D6, 0x30C3, 0x30B7, 0x30A7, 0x30EB),
                    0x3335 => array(0x30D5, 0x30E9, 0x30F3), 0x3336 => array(0x30D8, 0x30AF, 0x30BF, 0x30FC, 0x30EB), 0x3337 => array(0x30DA, 0x30BD),
                    0x3338 => array(0x30DA, 0x30CB, 0x30D2), 0x3339 => array(0x30D8, 0x30EB, 0x30C4), 0x333A => array(0x30DA, 0x30F3, 0x30B9),
                    0x333B => array(0x30DA, 0x30FC, 0x30B8), 0x333C => array(0x30D9, 0x30FC, 0x30BF), 0x333D => array(0x30DD, 0x30A4, 0x30F3, 0x30C8),
                    0x333E => array(0x30DC, 0x30EB, 0x30C8), 0x333F => array(0x30DB, 0x30F3), 0x3340 => array(0x30DD, 0x30F3, 0x30C9),
                    0x3341 => array(0x30DB, 0x30FC, 0x30EB), 0x3342 => array(0x30DB, 0x30FC, 0x30F3), 0x3343 => array(0x30DE, 0x30A4, 0x30AF, 0x30ED),
                    0x3344 => array(0x30DE, 0x30A4, 0x30EB), 0x3345 => array(0x30DE, 0x30C3, 0x30CF), 0x3346 => array(0x30DE, 0x30EB, 0x30AF),
                    0x3347 => array(0x30DE, 0x30F3, 0x30B7, 0x30E7, 0x30F3), 0x3348 => array(0x30DF, 0x30AF, 0x30ED, 0x30F3), 0x3349 => array(0x30DF, 0x30EA),
                    0x334A => array(0x30DF, 0x30EA, 0x30D0, 0x30FC, 0x30EB), 0x334B => array(0x30E1, 0x30AC), 0x334C => array(0x30E1, 0x30AC, 0x30C8, 0x30F3),
                    0x334D => array(0x30E1, 0x30FC, 0x30C8, 0x30EB), 0x334E => array(0x30E4, 0x30FC, 0x30C9), 0x334F => array(0x30E4, 0x30FC, 0x30EB),
                    0x3350 => array(0x30E6, 0x30A2, 0x30F3), 0x3351 => array(0x30EA, 0x30C3, 0x30C8, 0x30EB), 0x3352 => array(0x30EA, 0x30E9),
                    0x3353 => array(0x30EB, 0x30D4, 0x30FC), 0x3354 => array(0x30EB, 0x30FC, 0x30D6, 0x30EB), 0x3355 => array(0x30EC, 0x30E0),
                    0x3356 => array(0x30EC, 0x30F3, 0x30C8, 0x30B2, 0x30F3), 0x3357 => array(0x30EF, 0x30C3, 0x30C8), 0x3358 => array(0x30, 0x70B9),
                    0x3359 => array(0x31, 0x70B9), 0x335A => array(0x32, 0x70B9), 0x335B => array(0x33, 0x70B9),
                    0x335C => array(0x34, 0x70B9), 0x335D => array(0x35, 0x70B9), 0x335E => array(0x36, 0x70B9),
                    0x335F => array(0x37, 0x70B9), 0x3360 => array(0x38, 0x70B9), 0x3361 => array(0x39, 0x70B9),
                    0x3362 => array(0x31, 0x30, 0x70B9), 0x3363 => array(0x31, 0x31, 0x70B9), 0x3364 => array(0x31, 0x32, 0x70B9),
                    0x3365 => array(0x31, 0x33, 0x70B9), 0x3366 => array(0x31, 0x34, 0x70B9), 0x3367 => array(0x31, 0x35, 0x70B9),
                    0x3368 => array(0x31, 0x36, 0x70B9), 0x3369 => array(0x31, 0x37, 0x70B9), 0x336A => array(0x31, 0x38, 0x70B9),
                    0x336B => array(0x31, 0x39, 0x70B9), 0x336C => array(0x32, 0x30, 0x70B9), 0x336D => array(0x32, 0x31, 0x70B9),
                    0x336E => array(0x32, 0x32, 0x70B9), 0x336F => array(0x32, 0x33, 0x70B9), 0x3370 => array(0x32, 0x34, 0x70B9),
                    0x3371 => array(0x68, 0x70, 0x61), 0x3372 => array(0x64, 0x61), 0x3373 => array(0x61, 0x75),
                    0x3374 => array(0x62, 0x61, 0x72), 0x3375 => array(0x6F, 0x76), 0x3376 => array(0x70, 0x63),
                    0x3377 => array(0x64, 0x6D), 0x3378 => array(0x64, 0x6D, 0x32), 0x3379 => array(0x64, 0x6D, 0x33),
                    0x337A => array(0x69, 0x75), 0x337B => array(0x5E73, 0x6210), 0x337C => array(0x662D, 0x548C),
                    0x337D => array(0x5927, 0x6B63), 0x337E => array(0x660E, 0x6CBB), 0x337F => array(0x682A, 0x5F0F, 0x4F1A, 0x793E),
                    0x3380 => array(0x70, 0x61), 0x3381 => array(0x6E, 0x61), 0x3382 => array(0x3BC, 0x61),
                    0x3383 => array(0x6D, 0x61), 0x3384 => array(0x6B, 0x61), 0x3385 => array(0x6B, 0x62),
                    0x3386 => array(0x6D, 0x62), 0x3387 => array(0x67, 0x62), 0x3388 => array(0x63, 0x61, 0x6C),
                    0x3389 => array(0x6B, 0x63, 0x61, 0x6C), 0x338A => array(0x70, 0x66), 0x338B => array(0x6E, 0x66),
                    0x338C => array(0x3BC, 0x66), 0x338D => array(0x3BC, 0x67), 0x338E => array(0x6D, 0x67),
                    0x338F => array(0x6B, 0x67), 0x3390 => array(0x68, 0x7A), 0x3391 => array(0x6B, 0x68, 0x7A),
                    0x3392 => array(0x6D, 0x68, 0x7A), 0x3393 => array(0x67, 0x68, 0x7A), 0x3394 => array(0x74, 0x68, 0x7A),
                    0x3395 => array(0x3BC, 0x6C), 0x3396 => array(0x6D, 0x6C), 0x3397 => array(0x64, 0x6C),
                    0x3398 => array(0x6B, 0x6C), 0x3399 => array(0x66, 0x6D), 0x339A => array(0x6E, 0x6D),
                    0x339B => array(0x3BC, 0x6D), 0x339C => array(0x6D, 0x6D), 0x339D => array(0x63, 0x6D),
                    0x339E => array(0x6B, 0x6D), 0x339F => array(0x6D, 0x6D, 0x32), 0x33A0 => array(0x63, 0x6D, 0x32),
                    0x33A1 => array(0x6D, 0x32), 0x33A2 => array(0x6B, 0x6D, 0x32), 0x33A3 => array(0x6D, 0x6D, 0x33),
                    0x33A4 => array(0x63, 0x6D, 0x33), 0x33A5 => array(0x6D, 0x33), 0x33A6 => array(0x6B, 0x6D, 0x33),
                    0x33A7 => array(0x6D, 0x2215, 0x73), 0x33A8 => array(0x6D, 0x2215, 0x73, 0x32), 0x33A9 => array(0x70, 0x61),
                    0x33AA => array(0x6B, 0x70, 0x61), 0x33AB => array(0x6D, 0x70, 0x61), 0x33AC => array(0x67, 0x70, 0x61),
                    0x33AD => array(0x72, 0x61, 0x64), 0x33AE => array(0x72, 0x61, 0x64, 0x2215, 0x73), 0x33AF => array(0x72, 0x61, 0x64, 0x2215, 0x73, 0x32),
                    0x33B0 => array(0x70, 0x73), 0x33B1 => array(0x6E, 0x73), 0x33B2 => array(0x3BC, 0x73),
                    0x33B3 => array(0x6D, 0x73), 0x33B4 => array(0x70, 0x76), 0x33B5 => array(0x6E, 0x76),
                    0x33B6 => array(0x3BC, 0x76), 0x33B7 => array(0x6D, 0x76), 0x33B8 => array(0x6B, 0x76),
                    0x33B9 => array(0x6D, 0x76), 0x33BA => array(0x70, 0x77), 0x33BB => array(0x6E, 0x77),
                    0x33BC => array(0x3BC, 0x77), 0x33BD => array(0x6D, 0x77), 0x33BE => array(0x6B, 0x77),
                    0x33BF => array(0x6D, 0x77), 0x33C0 => array(0x6B, 0x3C9), 0x33C1 => array(0x6D, 0x3C9),
                    0x33C3 => array(0x62, 0x71), 0x33C4 => array(0x63, 0x63), 0x33C5 => array(0x63, 0x64),
                    0x33C6 => array(0x63, 0x2215, 0x6B, 0x67), 0x33C8 => array(0x64, 0x62), 0x33C9 => array(0x67, 0x79),
                    0x33CA => array(0x68, 0x61), 0x33CB => array(0x68, 0x70), 0x33CC => array(0x69, 0x6E),
                    0x33CD => array(0x6B, 0x6B), 0x33CE => array(0x6B, 0x6D), 0x33CF => array(0x6B, 0x74),
                    0x33D0 => array(0x6C, 0x6D), 0x33D1 => array(0x6C, 0x6E), 0x33D2 => array(0x6C, 0x6F, 0x67),
                    0x33D3 => array(0x6C, 0x78), 0x33D4 => array(0x6D, 0x62), 0x33D5 => array(0x6D, 0x69, 0x6C),
                    0x33D6 => array(0x6D, 0x6F, 0x6C), 0x33D7 => array(0x70, 0x68), 0x33D9 => array(0x70, 0x70, 0x6D),
                    0x33DA => array(0x70, 0x72), 0x33DB => array(0x73, 0x72), 0x33DC => array(0x73, 0x76),
                    0x33DD => array(0x77, 0x62), 0x33DE => array(0x76, 0x2215, 0x6D), 0x33DF => array(0x61, 0x2215, 0x6D),
                    0x33E0 => array(0x31, 0x65E5), 0x33E1 => array(0x32, 0x65E5), 0x33E2 => array(0x33, 0x65E5),
                    0x33E3 => array(0x34, 0x65E5), 0x33E4 => array(0x35, 0x65E5), 0x33E5 => array(0x36, 0x65E5),
                    0x33E6 => array(0x37, 0x65E5), 0x33E7 => array(0x38, 0x65E5), 0x33E8 => array(0x39, 0x65E5),
                    0x33E9 => array(0x31, 0x30, 0x65E5), 0x33EA => array(0x31, 0x31, 0x65E5), 0x33EB => array(0x31, 0x32, 0x65E5),
                    0x33EC => array(0x31, 0x33, 0x65E5), 0x33ED => array(0x31, 0x34, 0x65E5), 0x33EE => array(0x31, 0x35, 0x65E5),
                    0x33EF => array(0x31, 0x36, 0x65E5), 0x33F0 => array(0x31, 0x37, 0x65E5), 0x33F1 => array(0x31, 0x38, 0x65E5),
                    0x33F2 => array(0x31, 0x39, 0x65E5), 0x33F3 => array(0x32, 0x30, 0x65E5), 0x33F4 => array(0x32, 0x31, 0x65E5),
                    0x33F5 => array(0x32, 0x32, 0x65E5), 0x33F6 => array(0x32, 0x33, 0x65E5), 0x33F7 => array(0x32, 0x34, 0x65E5),
                    0x33F8 => array(0x32, 0x35, 0x65E5), 0x33F9 => array(0x32, 0x36, 0x65E5), 0x33FA => array(0x32, 0x37, 0x65E5),
                    0x33FB => array(0x32, 0x38, 0x65E5), 0x33FC => array(0x32, 0x39, 0x65E5), 0x33FD => array(0x33, 0x30, 0x65E5),
                    0x33FE => array(0x33, 0x31, 0x65E5), 0x33FF => array(0x67, 0x61, 0x6C), 0xA640 => array(0xA641),
                    0xA642 => array(0xA643), 0xA644 => array(0xA645), 0xA646 => array(0xA647),
                    0xA648 => array(0xA649), 0xA64A => array(0xA64B), 0xA64C => array(0xA64D),
                    0xA64E => array(0xA64F), 0xA650 => array(0xA651), 0xA652 => array(0xA653),
                    0xA654 => array(0xA655), 0xA656 => array(0xA657), 0xA658 => array(0xA659),
                    0xA65A => array(0xA65B), 0xA65C => array(0xA65D), 0xA65E => array(0xA65F),
                    0xA660 => array(0xA661), 0xA662 => array(0xA663), 0xA664 => array(0xA665),
                    0xA666 => array(0xA667), 0xA668 => array(0xA669), 0xA66A => array(0xA66B),
                    0xA66C => array(0xA66D), 0xA680 => array(0xA681), 0xA682 => array(0xA683),
                    0xA684 => array(0xA685), 0xA686 => array(0xA687), 0xA688 => array(0xA689),
                    0xA68A => array(0xA68B), 0xA68C => array(0xA68D), 0xA68E => array(0xA68F),
                    0xA690 => array(0xA691), 0xA692 => array(0xA693), 0xA694 => array(0xA695),
                    0xA696 => array(0xA697), 0xA698 => array(0xA699), 0xA69A => array(0xA69B),
                    0xA69C => array(0x44A), 0xA69D => array(0x44C), 0xA722 => array(0xA723),
                    0xA724 => array(0xA725), 0xA726 => array(0xA727), 0xA728 => array(0xA729),
                    0xA72A => array(0xA72B), 0xA72C => array(0xA72D), 0xA72E => array(0xA72F),
                    0xA732 => array(0xA733), 0xA734 => array(0xA735), 0xA736 => array(0xA737),
                    0xA738 => array(0xA739), 0xA73A => array(0xA73B), 0xA73C => array(0xA73D),
                    0xA73E => array(0xA73F), 0xA740 => array(0xA741), 0xA742 => array(0xA743),
                    0xA744 => array(0xA745), 0xA746 => array(0xA747), 0xA748 => array(0xA749),
                    0xA74A => array(0xA74B), 0xA74C => array(0xA74D), 0xA74E => array(0xA74F),
                    0xA750 => array(0xA751), 0xA752 => array(0xA753), 0xA754 => array(0xA755),
                    0xA756 => array(0xA757), 0xA758 => array(0xA759), 0xA75A => array(0xA75B),
                    0xA75C => array(0xA75D), 0xA75E => array(0xA75F), 0xA760 => array(0xA761),
                    0xA762 => array(0xA763), 0xA764 => array(0xA765), 0xA766 => array(0xA767),
                    0xA768 => array(0xA769), 0xA76A => array(0xA76B), 0xA76C => array(0xA76D),
                    0xA76E => array(0xA76F), 0xA770 => array(0xA76F), 0xA779 => array(0xA77A),
                    0xA77B => array(0xA77C), 0xA77D => array(0x1D79), 0xA77E => array(0xA77F),
                    0xA780 => array(0xA781), 0xA782 => array(0xA783), 0xA784 => array(0xA785),
                    0xA786 => array(0xA787), 0xA78B => array(0xA78C), 0xA78D => array(0x265),
                    0xA790 => array(0xA791), 0xA792 => array(0xA793), 0xA796 => array(0xA797),
                    0xA798 => array(0xA799), 0xA79A => array(0xA79B), 0xA79C => array(0xA79D),
                    0xA79E => array(0xA79F), 0xA7A0 => array(0xA7A1), 0xA7A2 => array(0xA7A3),
                    0xA7A4 => array(0xA7A5), 0xA7A6 => array(0xA7A7), 0xA7A8 => array(0xA7A9),
                    0xA7AA => array(0x266), 0xA7AB => array(0x25C), 0xA7AC => array(0x261),
                    0xA7AD => array(0x26C), 0xA7B0 => array(0x29E), 0xA7B1 => array(0x287),
                    0xA7F8 => array(0x127), 0xA7F9 => array(0x153), 0xAB5C => array(0xA727),
                    0xAB5D => array(0xAB37), 0xAB5E => array(0x26B), 0xAB5F => array(0xAB52),
                    0xF900 => array(0x8C48), 0xF901 => array(0x66F4), 0xF902 => array(0x8ECA),
                    0xF903 => array(0x8CC8), 0xF904 => array(0x6ED1), 0xF905 => array(0x4E32),
                    0xF906 => array(0x53E5), 0xF907 => array(0x9F9C), 0xF908 => array(0x9F9C),
                    0xF909 => array(0x5951), 0xF90A => array(0x91D1), 0xF90B => array(0x5587),
                    0xF90C => array(0x5948), 0xF90D => array(0x61F6), 0xF90E => array(0x7669),
                    0xF90F => array(0x7F85), 0xF910 => array(0x863F), 0xF911 => array(0x87BA),
                    0xF912 => array(0x88F8), 0xF913 => array(0x908F), 0xF914 => array(0x6A02),
                    0xF915 => array(0x6D1B), 0xF916 => array(0x70D9), 0xF917 => array(0x73DE),
                    0xF918 => array(0x843D), 0xF919 => array(0x916A), 0xF91A => array(0x99F1),
                    0xF91B => array(0x4E82), 0xF91C => array(0x5375), 0xF91D => array(0x6B04),
                    0xF91E => array(0x721B), 0xF91F => array(0x862D), 0xF920 => array(0x9E1E),
                    0xF921 => array(0x5D50), 0xF922 => array(0x6FEB), 0xF923 => array(0x85CD),
                    0xF924 => array(0x8964), 0xF925 => array(0x62C9), 0xF926 => array(0x81D8),
                    0xF927 => array(0x881F), 0xF928 => array(0x5ECA), 0xF929 => array(0x6717),
                    0xF92A => array(0x6D6A), 0xF92B => array(0x72FC), 0xF92C => array(0x90CE),
                    0xF92D => array(0x4F86), 0xF92E => array(0x51B7), 0xF92F => array(0x52DE),
                    0xF930 => array(0x64C4), 0xF931 => array(0x6AD3), 0xF932 => array(0x7210),
                    0xF933 => array(0x76E7), 0xF934 => array(0x8001), 0xF935 => array(0x8606),
                    0xF936 => array(0x865C), 0xF937 => array(0x8DEF), 0xF938 => array(0x9732),
                    0xF939 => array(0x9B6F), 0xF93A => array(0x9DFA), 0xF93B => array(0x788C),
                    0xF93C => array(0x797F), 0xF93D => array(0x7DA0), 0xF93E => array(0x83C9),
                    0xF93F => array(0x9304), 0xF940 => array(0x9E7F), 0xF941 => array(0x8AD6),
                    0xF942 => array(0x58DF), 0xF943 => array(0x5F04), 0xF944 => array(0x7C60),
                    0xF945 => array(0x807E), 0xF946 => array(0x7262), 0xF947 => array(0x78CA),
                    0xF948 => array(0x8CC2), 0xF949 => array(0x96F7), 0xF94A => array(0x58D8),
                    0xF94B => array(0x5C62), 0xF94C => array(0x6A13), 0xF94D => array(0x6DDA),
                    0xF94E => array(0x6F0F), 0xF94F => array(0x7D2F), 0xF950 => array(0x7E37),
                    0xF951 => array(0x964B), 0xF952 => array(0x52D2), 0xF953 => array(0x808B),
                    0xF954 => array(0x51DC), 0xF955 => array(0x51CC), 0xF956 => array(0x7A1C),
                    0xF957 => array(0x7DBE), 0xF958 => array(0x83F1), 0xF959 => array(0x9675),
                    0xF95A => array(0x8B80), 0xF95B => array(0x62CF), 0xF95C => array(0x6A02),
                    0xF95D => array(0x8AFE), 0xF95E => array(0x4E39), 0xF95F => array(0x5BE7),
                    0xF960 => array(0x6012), 0xF961 => array(0x7387), 0xF962 => array(0x7570),
                    0xF963 => array(0x5317), 0xF964 => array(0x78FB), 0xF965 => array(0x4FBF),
                    0xF966 => array(0x5FA9), 0xF967 => array(0x4E0D), 0xF968 => array(0x6CCC),
                    0xF969 => array(0x6578), 0xF96A => array(0x7D22), 0xF96B => array(0x53C3),
                    0xF96C => array(0x585E), 0xF96D => array(0x7701), 0xF96E => array(0x8449),
                    0xF96F => array(0x8AAA), 0xF970 => array(0x6BBA), 0xF971 => array(0x8FB0),
                    0xF972 => array(0x6C88), 0xF973 => array(0x62FE), 0xF974 => array(0x82E5),
                    0xF975 => array(0x63A0), 0xF976 => array(0x7565), 0xF977 => array(0x4EAE),
                    0xF978 => array(0x5169), 0xF979 => array(0x51C9), 0xF97A => array(0x6881),
                    0xF97B => array(0x7CE7), 0xF97C => array(0x826F), 0xF97D => array(0x8AD2),
                    0xF97E => array(0x91CF), 0xF97F => array(0x52F5), 0xF980 => array(0x5442),
                    0xF981 => array(0x5973), 0xF982 => array(0x5EEC), 0xF983 => array(0x65C5),
                    0xF984 => array(0x6FFE), 0xF985 => array(0x792A), 0xF986 => array(0x95AD),
                    0xF987 => array(0x9A6A), 0xF988 => array(0x9E97), 0xF989 => array(0x9ECE),
                    0xF98A => array(0x529B), 0xF98B => array(0x66C6), 0xF98C => array(0x6B77),
                    0xF98D => array(0x8F62), 0xF98E => array(0x5E74), 0xF98F => array(0x6190),
                    0xF990 => array(0x6200), 0xF991 => array(0x649A), 0xF992 => array(0x6F23),
                    0xF993 => array(0x7149), 0xF994 => array(0x7489), 0xF995 => array(0x79CA),
                    0xF996 => array(0x7DF4), 0xF997 => array(0x806F), 0xF998 => array(0x8F26),
                    0xF999 => array(0x84EE), 0xF99A => array(0x9023), 0xF99B => array(0x934A),
                    0xF99C => array(0x5217), 0xF99D => array(0x52A3), 0xF99E => array(0x54BD),
                    0xF99F => array(0x70C8), 0xF9A0 => array(0x88C2), 0xF9A1 => array(0x8AAA),
                    0xF9A2 => array(0x5EC9), 0xF9A3 => array(0x5FF5), 0xF9A4 => array(0x637B),
                    0xF9A5 => array(0x6BAE), 0xF9A6 => array(0x7C3E), 0xF9A7 => array(0x7375),
                    0xF9A8 => array(0x4EE4), 0xF9A9 => array(0x56F9), 0xF9AA => array(0x5BE7),
                    0xF9AB => array(0x5DBA), 0xF9AC => array(0x601C), 0xF9AD => array(0x73B2),
                    0xF9AE => array(0x7469), 0xF9AF => array(0x7F9A), 0xF9B0 => array(0x8046),
                    0xF9B1 => array(0x9234), 0xF9B2 => array(0x96F6), 0xF9B3 => array(0x9748),
                    0xF9B4 => array(0x9818), 0xF9B5 => array(0x4F8B), 0xF9B6 => array(0x79AE),
                    0xF9B7 => array(0x91B4), 0xF9B8 => array(0x96B8), 0xF9B9 => array(0x60E1),
                    0xF9BA => array(0x4E86), 0xF9BB => array(0x50DA), 0xF9BC => array(0x5BEE),
                    0xF9BD => array(0x5C3F), 0xF9BE => array(0x6599), 0xF9BF => array(0x6A02),
                    0xF9C0 => array(0x71CE), 0xF9C1 => array(0x7642), 0xF9C2 => array(0x84FC),
                    0xF9C3 => array(0x907C), 0xF9C4 => array(0x9F8D), 0xF9C5 => array(0x6688),
                    0xF9C6 => array(0x962E), 0xF9C7 => array(0x5289), 0xF9C8 => array(0x677B),
                    0xF9C9 => array(0x67F3), 0xF9CA => array(0x6D41), 0xF9CB => array(0x6E9C),
                    0xF9CC => array(0x7409), 0xF9CD => array(0x7559), 0xF9CE => array(0x786B),
                    0xF9CF => array(0x7D10), 0xF9D0 => array(0x985E), 0xF9D1 => array(0x516D),
                    0xF9D2 => array(0x622E), 0xF9D3 => array(0x9678), 0xF9D4 => array(0x502B),
                    0xF9D5 => array(0x5D19), 0xF9D6 => array(0x6DEA), 0xF9D7 => array(0x8F2A),
                    0xF9D8 => array(0x5F8B), 0xF9D9 => array(0x6144), 0xF9DA => array(0x6817),
                    0xF9DB => array(0x7387), 0xF9DC => array(0x9686), 0xF9DD => array(0x5229),
                    0xF9DE => array(0x540F), 0xF9DF => array(0x5C65), 0xF9E0 => array(0x6613),
                    0xF9E1 => array(0x674E), 0xF9E2 => array(0x68A8), 0xF9E3 => array(0x6CE5),
                    0xF9E4 => array(0x7406), 0xF9E5 => array(0x75E2), 0xF9E6 => array(0x7F79),
                    0xF9E7 => array(0x88CF), 0xF9E8 => array(0x88E1), 0xF9E9 => array(0x91CC),
                    0xF9EA => array(0x96E2), 0xF9EB => array(0x533F), 0xF9EC => array(0x6EBA),
                    0xF9ED => array(0x541D), 0xF9EE => array(0x71D0), 0xF9EF => array(0x7498),
                    0xF9F0 => array(0x85FA), 0xF9F1 => array(0x96A3), 0xF9F2 => array(0x9C57),
                    0xF9F3 => array(0x9E9F), 0xF9F4 => array(0x6797), 0xF9F5 => array(0x6DCB),
                    0xF9F6 => array(0x81E8), 0xF9F7 => array(0x7ACB), 0xF9F8 => array(0x7B20),
                    0xF9F9 => array(0x7C92), 0xF9FA => array(0x72C0), 0xF9FB => array(0x7099),
                    0xF9FC => array(0x8B58), 0xF9FD => array(0x4EC0), 0xF9FE => array(0x8336),
                    0xF9FF => array(0x523A), 0xFA00 => array(0x5207), 0xFA01 => array(0x5EA6),
                    0xFA02 => array(0x62D3), 0xFA03 => array(0x7CD6), 0xFA04 => array(0x5B85),
                    0xFA05 => array(0x6D1E), 0xFA06 => array(0x66B4), 0xFA07 => array(0x8F3B),
                    0xFA08 => array(0x884C), 0xFA09 => array(0x964D), 0xFA0A => array(0x898B),
                    0xFA0B => array(0x5ED3), 0xFA0C => array(0x5140), 0xFA0D => array(0x55C0),
                    0xFA10 => array(0x585A), 0xFA12 => array(0x6674), 0xFA15 => array(0x51DE),
                    0xFA16 => array(0x732A), 0xFA17 => array(0x76CA), 0xFA18 => array(0x793C),
                    0xFA19 => array(0x795E), 0xFA1A => array(0x7965), 0xFA1B => array(0x798F),
                    0xFA1C => array(0x9756), 0xFA1D => array(0x7CBE), 0xFA1E => array(0x7FBD),
                    0xFA20 => array(0x8612), 0xFA22 => array(0x8AF8), 0xFA25 => array(0x9038),
                    0xFA26 => array(0x90FD), 0xFA2A => array(0x98EF), 0xFA2B => array(0x98FC),
                    0xFA2C => array(0x9928), 0xFA2D => array(0x9DB4), 0xFA2E => array(0x90DE),
                    0xFA2F => array(0x96B7), 0xFA30 => array(0x4FAE), 0xFA31 => array(0x50E7),
                    0xFA32 => array(0x514D), 0xFA33 => array(0x52C9), 0xFA34 => array(0x52E4),
                    0xFA35 => array(0x5351), 0xFA36 => array(0x559D), 0xFA37 => array(0x5606),
                    0xFA38 => array(0x5668), 0xFA39 => array(0x5840), 0xFA3A => array(0x58A8),
                    0xFA3B => array(0x5C64), 0xFA3C => array(0x5C6E), 0xFA3D => array(0x6094),
                    0xFA3E => array(0x6168), 0xFA3F => array(0x618E), 0xFA40 => array(0x61F2),
                    0xFA41 => array(0x654F), 0xFA42 => array(0x65E2), 0xFA43 => array(0x6691),
                    0xFA44 => array(0x6885), 0xFA45 => array(0x6D77), 0xFA46 => array(0x6E1A),
                    0xFA47 => array(0x6F22), 0xFA48 => array(0x716E), 0xFA49 => array(0x722B),
                    0xFA4A => array(0x7422), 0xFA4B => array(0x7891), 0xFA4C => array(0x793E),
                    0xFA4D => array(0x7949), 0xFA4E => array(0x7948), 0xFA4F => array(0x7950),
                    0xFA50 => array(0x7956), 0xFA51 => array(0x795D), 0xFA52 => array(0x798D),
                    0xFA53 => array(0x798E), 0xFA54 => array(0x7A40), 0xFA55 => array(0x7A81),
                    0xFA56 => array(0x7BC0), 0xFA57 => array(0x7DF4), 0xFA58 => array(0x7E09),
                    0xFA59 => array(0x7E41), 0xFA5A => array(0x7F72), 0xFA5B => array(0x8005),
                    0xFA5C => array(0x81ED), 0xFA5D => array(0x8279), 0xFA5E => array(0x8279),
                    0xFA5F => array(0x8457), 0xFA60 => array(0x8910), 0xFA61 => array(0x8996),
                    0xFA62 => array(0x8B01), 0xFA63 => array(0x8B39), 0xFA64 => array(0x8CD3),
                    0xFA65 => array(0x8D08), 0xFA66 => array(0x8FB6), 0xFA67 => array(0x9038),
                    0xFA68 => array(0x96E3), 0xFA69 => array(0x97FF), 0xFA6A => array(0x983B),
                    0xFA6B => array(0x6075), 0xFA6C => array(0x242EE), 0xFA6D => array(0x8218),
                    0xFA70 => array(0x4E26), 0xFA71 => array(0x51B5), 0xFA72 => array(0x5168),
                    0xFA73 => array(0x4F80), 0xFA74 => array(0x5145), 0xFA75 => array(0x5180),
                    0xFA76 => array(0x52C7), 0xFA77 => array(0x52FA), 0xFA78 => array(0x559D),
                    0xFA79 => array(0x5555), 0xFA7A => array(0x5599), 0xFA7B => array(0x55E2),
                    0xFA7C => array(0x585A), 0xFA7D => array(0x58B3), 0xFA7E => array(0x5944),
                    0xFA7F => array(0x5954), 0xFA80 => array(0x5A62), 0xFA81 => array(0x5B28),
                    0xFA82 => array(0x5ED2), 0xFA83 => array(0x5ED9), 0xFA84 => array(0x5F69),
                    0xFA85 => array(0x5FAD), 0xFA86 => array(0x60D8), 0xFA87 => array(0x614E),
                    0xFA88 => array(0x6108), 0xFA89 => array(0x618E), 0xFA8A => array(0x6160),
                    0xFA8B => array(0x61F2), 0xFA8C => array(0x6234), 0xFA8D => array(0x63C4),
                    0xFA8E => array(0x641C), 0xFA8F => array(0x6452), 0xFA90 => array(0x6556),
                    0xFA91 => array(0x6674), 0xFA92 => array(0x6717), 0xFA93 => array(0x671B),
                    0xFA94 => array(0x6756), 0xFA95 => array(0x6B79), 0xFA96 => array(0x6BBA),
                    0xFA97 => array(0x6D41), 0xFA98 => array(0x6EDB), 0xFA99 => array(0x6ECB),
                    0xFA9A => array(0x6F22), 0xFA9B => array(0x701E), 0xFA9C => array(0x716E),
                    0xFA9D => array(0x77A7), 0xFA9E => array(0x7235), 0xFA9F => array(0x72AF),
                    0xFAA0 => array(0x732A), 0xFAA1 => array(0x7471), 0xFAA2 => array(0x7506),
                    0xFAA3 => array(0x753B), 0xFAA4 => array(0x761D), 0xFAA5 => array(0x761F),
                    0xFAA6 => array(0x76CA), 0xFAA7 => array(0x76DB), 0xFAA8 => array(0x76F4),
                    0xFAA9 => array(0x774A), 0xFAAA => array(0x7740), 0xFAAB => array(0x78CC),
                    0xFAAC => array(0x7AB1), 0xFAAD => array(0x7BC0), 0xFAAE => array(0x7C7B),
                    0xFAAF => array(0x7D5B), 0xFAB0 => array(0x7DF4), 0xFAB1 => array(0x7F3E),
                    0xFAB2 => array(0x8005), 0xFAB3 => array(0x8352), 0xFAB4 => array(0x83EF),
                    0xFAB5 => array(0x8779), 0xFAB6 => array(0x8941), 0xFAB7 => array(0x8986),
                    0xFAB8 => array(0x8996), 0xFAB9 => array(0x8ABF), 0xFABA => array(0x8AF8),
                    0xFABB => array(0x8ACB), 0xFABC => array(0x8B01), 0xFABD => array(0x8AFE),
                    0xFABE => array(0x8AED), 0xFABF => array(0x8B39), 0xFAC0 => array(0x8B8A),
                    0xFAC1 => array(0x8D08), 0xFAC2 => array(0x8F38), 0xFAC3 => array(0x9072),
                    0xFAC4 => array(0x9199), 0xFAC5 => array(0x9276), 0xFAC6 => array(0x967C),
                    0xFAC7 => array(0x96E3), 0xFAC8 => array(0x9756), 0xFAC9 => array(0x97DB),
                    0xFACA => array(0x97FF), 0xFACB => array(0x980B), 0xFACC => array(0x983B),
                    0xFACD => array(0x9B12), 0xFACE => array(0x9F9C), 0xFACF => array(0x2284A),
                    0xFAD0 => array(0x22844), 0xFAD1 => array(0x233D5), 0xFAD2 => array(0x3B9D),
                    0xFAD3 => array(0x4018), 0xFAD4 => array(0x4039), 0xFAD5 => array(0x25249),
                    0xFAD6 => array(0x25CD0), 0xFAD7 => array(0x27ED3), 0xFAD8 => array(0x9F43),
                    0xFAD9 => array(0x9F8E), 0xFB00 => array(0x66, 0x66), 0xFB01 => array(0x66, 0x69),
                    0xFB02 => array(0x66, 0x6C), 0xFB03 => array(0x66, 0x66, 0x69), 0xFB04 => array(0x66, 0x66, 0x6C),
                    0xFB05 => array(0x73, 0x74), 0xFB06 => array(0x73, 0x74), 0xFB13 => array(0x574, 0x576),
                    0xFB14 => array(0x574, 0x565), 0xFB15 => array(0x574, 0x56B), 0xFB16 => array(0x57E, 0x576),
                    0xFB17 => array(0x574, 0x56D), 0xFB1D => array(0x5D9, 0x5B4), 0xFB1F => array(0x5F2, 0x5B7),
                    0xFB20 => array(0x5E2), 0xFB21 => array(0x5D0), 0xFB22 => array(0x5D3),
                    0xFB23 => array(0x5D4), 0xFB24 => array(0x5DB), 0xFB25 => array(0x5DC),
                    0xFB26 => array(0x5DD), 0xFB27 => array(0x5E8), 0xFB28 => array(0x5EA),
                    0xFB2A => array(0x5E9, 0x5C1), 0xFB2B => array(0x5E9, 0x5C2), 0xFB2C => array(0x5E9, 0x5BC, 0x5C1),
                    0xFB2D => array(0x5E9, 0x5BC, 0x5C2), 0xFB2E => array(0x5D0, 0x5B7), 0xFB2F => array(0x5D0, 0x5B8),
                    0xFB30 => array(0x5D0, 0x5BC), 0xFB31 => array(0x5D1, 0x5BC), 0xFB32 => array(0x5D2, 0x5BC),
                    0xFB33 => array(0x5D3, 0x5BC), 0xFB34 => array(0x5D4, 0x5BC), 0xFB35 => array(0x5D5, 0x5BC),
                    0xFB36 => array(0x5D6, 0x5BC), 0xFB38 => array(0x5D8, 0x5BC), 0xFB39 => array(0x5D9, 0x5BC),
                    0xFB3A => array(0x5DA, 0x5BC), 0xFB3B => array(0x5DB, 0x5BC), 0xFB3C => array(0x5DC, 0x5BC),
                    0xFB3E => array(0x5DE, 0x5BC), 0xFB40 => array(0x5E0, 0x5BC), 0xFB41 => array(0x5E1, 0x5BC),
                    0xFB43 => array(0x5E3, 0x5BC), 0xFB44 => array(0x5E4, 0x5BC), 0xFB46 => array(0x5E6, 0x5BC),
                    0xFB47 => array(0x5E7, 0x5BC), 0xFB48 => array(0x5E8, 0x5BC), 0xFB49 => array(0x5E9, 0x5BC),
                    0xFB4A => array(0x5EA, 0x5BC), 0xFB4B => array(0x5D5, 0x5B9), 0xFB4C => array(0x5D1, 0x5BF),
                    0xFB4D => array(0x5DB, 0x5BF), 0xFB4E => array(0x5E4, 0x5BF), 0xFB4F => array(0x5D0, 0x5DC),
                    0xFB50 => array(0x671), 0xFB51 => array(0x671), 0xFB52 => array(0x67B),
                    0xFB53 => array(0x67B), 0xFB54 => array(0x67B), 0xFB55 => array(0x67B),
                    0xFB56 => array(0x67E), 0xFB57 => array(0x67E), 0xFB58 => array(0x67E),
                    0xFB59 => array(0x67E), 0xFB5A => array(0x680), 0xFB5B => array(0x680),
                    0xFB5C => array(0x680), 0xFB5D => array(0x680), 0xFB5E => array(0x67A),
                    0xFB5F => array(0x67A), 0xFB60 => array(0x67A), 0xFB61 => array(0x67A),
                    0xFB62 => array(0x67F), 0xFB63 => array(0x67F), 0xFB64 => array(0x67F),
                    0xFB65 => array(0x67F), 0xFB66 => array(0x679), 0xFB67 => array(0x679),
                    0xFB68 => array(0x679), 0xFB69 => array(0x679), 0xFB6A => array(0x6A4),
                    0xFB6B => array(0x6A4), 0xFB6C => array(0x6A4), 0xFB6D => array(0x6A4),
                    0xFB6E => array(0x6A6), 0xFB6F => array(0x6A6), 0xFB70 => array(0x6A6),
                    0xFB71 => array(0x6A6), 0xFB72 => array(0x684), 0xFB73 => array(0x684),
                    0xFB74 => array(0x684), 0xFB75 => array(0x684), 0xFB76 => array(0x683),
                    0xFB77 => array(0x683), 0xFB78 => array(0x683), 0xFB79 => array(0x683),
                    0xFB7A => array(0x686), 0xFB7B => array(0x686), 0xFB7C => array(0x686),
                    0xFB7D => array(0x686), 0xFB7E => array(0x687), 0xFB7F => array(0x687),
                    0xFB80 => array(0x687), 0xFB81 => array(0x687), 0xFB82 => array(0x68D),
                    0xFB83 => array(0x68D), 0xFB84 => array(0x68C), 0xFB85 => array(0x68C),
                    0xFB86 => array(0x68E), 0xFB87 => array(0x68E), 0xFB88 => array(0x688),
                    0xFB89 => array(0x688), 0xFB8A => array(0x698), 0xFB8B => array(0x698),
                    0xFB8C => array(0x691), 0xFB8D => array(0x691), 0xFB8E => array(0x6A9),
                    0xFB8F => array(0x6A9), 0xFB90 => array(0x6A9), 0xFB91 => array(0x6A9),
                    0xFB92 => array(0x6AF), 0xFB93 => array(0x6AF), 0xFB94 => array(0x6AF),
                    0xFB95 => array(0x6AF), 0xFB96 => array(0x6B3), 0xFB97 => array(0x6B3),
                    0xFB98 => array(0x6B3), 0xFB99 => array(0x6B3), 0xFB9A => array(0x6B1),
                    0xFB9B => array(0x6B1), 0xFB9C => array(0x6B1), 0xFB9D => array(0x6B1),
                    0xFB9E => array(0x6BA), 0xFB9F => array(0x6BA), 0xFBA0 => array(0x6BB),
                    0xFBA1 => array(0x6BB), 0xFBA2 => array(0x6BB), 0xFBA3 => array(0x6BB),
                    0xFBA4 => array(0x6C0), 0xFBA5 => array(0x6C0), 0xFBA6 => array(0x6C1),
                    0xFBA7 => array(0x6C1), 0xFBA8 => array(0x6C1), 0xFBA9 => array(0x6C1),
                    0xFBAA => array(0x6BE), 0xFBAB => array(0x6BE), 0xFBAC => array(0x6BE),
                    0xFBAD => array(0x6BE), 0xFBAE => array(0x6D2), 0xFBAF => array(0x6D2),
                    0xFBB0 => array(0x6D3), 0xFBB1 => array(0x6D3), 0xFBD3 => array(0x6AD),
                    0xFBD4 => array(0x6AD), 0xFBD5 => array(0x6AD), 0xFBD6 => array(0x6AD),
                    0xFBD7 => array(0x6C7), 0xFBD8 => array(0x6C7), 0xFBD9 => array(0x6C6),
                    0xFBDA => array(0x6C6), 0xFBDB => array(0x6C8), 0xFBDC => array(0x6C8),
                    0xFBDD => array(0x6C7, 0x674), 0xFBDE => array(0x6CB), 0xFBDF => array(0x6CB),
                    0xFBE0 => array(0x6C5), 0xFBE1 => array(0x6C5), 0xFBE2 => array(0x6C9),
                    0xFBE3 => array(0x6C9), 0xFBE4 => array(0x6D0), 0xFBE5 => array(0x6D0),
                    0xFBE6 => array(0x6D0), 0xFBE7 => array(0x6D0), 0xFBE8 => array(0x649),
                    0xFBE9 => array(0x649), 0xFBEA => array(0x626, 0x627), 0xFBEB => array(0x626, 0x627),
                    0xFBEC => array(0x626, 0x6D5), 0xFBED => array(0x626, 0x6D5), 0xFBEE => array(0x626, 0x648),
                    0xFBEF => array(0x626, 0x648), 0xFBF0 => array(0x626, 0x6C7), 0xFBF1 => array(0x626, 0x6C7),
                    0xFBF2 => array(0x626, 0x6C6), 0xFBF3 => array(0x626, 0x6C6), 0xFBF4 => array(0x626, 0x6C8),
                    0xFBF5 => array(0x626, 0x6C8), 0xFBF6 => array(0x626, 0x6D0), 0xFBF7 => array(0x626, 0x6D0),
                    0xFBF8 => array(0x626, 0x6D0), 0xFBF9 => array(0x626, 0x649), 0xFBFA => array(0x626, 0x649),
                    0xFBFB => array(0x626, 0x649), 0xFBFC => array(0x6CC), 0xFBFD => array(0x6CC),
                    0xFBFE => array(0x6CC), 0xFBFF => array(0x6CC), 0xFC00 => array(0x626, 0x62C),
                    0xFC01 => array(0x626, 0x62D), 0xFC02 => array(0x626, 0x645), 0xFC03 => array(0x626, 0x649),
                    0xFC04 => array(0x626, 0x64A), 0xFC05 => array(0x628, 0x62C), 0xFC06 => array(0x628, 0x62D),
                    0xFC07 => array(0x628, 0x62E), 0xFC08 => array(0x628, 0x645), 0xFC09 => array(0x628, 0x649),
                    0xFC0A => array(0x628, 0x64A), 0xFC0B => array(0x62A, 0x62C), 0xFC0C => array(0x62A, 0x62D),
                    0xFC0D => array(0x62A, 0x62E), 0xFC0E => array(0x62A, 0x645), 0xFC0F => array(0x62A, 0x649),
                    0xFC10 => array(0x62A, 0x64A), 0xFC11 => array(0x62B, 0x62C), 0xFC12 => array(0x62B, 0x645),
                    0xFC13 => array(0x62B, 0x649), 0xFC14 => array(0x62B, 0x64A), 0xFC15 => array(0x62C, 0x62D),
                    0xFC16 => array(0x62C, 0x645), 0xFC17 => array(0x62D, 0x62C), 0xFC18 => array(0x62D, 0x645),
                    0xFC19 => array(0x62E, 0x62C), 0xFC1A => array(0x62E, 0x62D), 0xFC1B => array(0x62E, 0x645),
                    0xFC1C => array(0x633, 0x62C), 0xFC1D => array(0x633, 0x62D), 0xFC1E => array(0x633, 0x62E),
                    0xFC1F => array(0x633, 0x645), 0xFC20 => array(0x635, 0x62D), 0xFC21 => array(0x635, 0x645),
                    0xFC22 => array(0x636, 0x62C), 0xFC23 => array(0x636, 0x62D), 0xFC24 => array(0x636, 0x62E),
                    0xFC25 => array(0x636, 0x645), 0xFC26 => array(0x637, 0x62D), 0xFC27 => array(0x637, 0x645),
                    0xFC28 => array(0x638, 0x645), 0xFC29 => array(0x639, 0x62C), 0xFC2A => array(0x639, 0x645),
                    0xFC2B => array(0x63A, 0x62C), 0xFC2C => array(0x63A, 0x645), 0xFC2D => array(0x641, 0x62C),
                    0xFC2E => array(0x641, 0x62D), 0xFC2F => array(0x641, 0x62E), 0xFC30 => array(0x641, 0x645),
                    0xFC31 => array(0x641, 0x649), 0xFC32 => array(0x641, 0x64A), 0xFC33 => array(0x642, 0x62D),
                    0xFC34 => array(0x642, 0x645), 0xFC35 => array(0x642, 0x649), 0xFC36 => array(0x642, 0x64A),
                    0xFC37 => array(0x643, 0x627), 0xFC38 => array(0x643, 0x62C), 0xFC39 => array(0x643, 0x62D),
                    0xFC3A => array(0x643, 0x62E), 0xFC3B => array(0x643, 0x644), 0xFC3C => array(0x643, 0x645),
                    0xFC3D => array(0x643, 0x649), 0xFC3E => array(0x643, 0x64A), 0xFC3F => array(0x644, 0x62C),
                    0xFC40 => array(0x644, 0x62D), 0xFC41 => array(0x644, 0x62E), 0xFC42 => array(0x644, 0x645),
                    0xFC43 => array(0x644, 0x649), 0xFC44 => array(0x644, 0x64A), 0xFC45 => array(0x645, 0x62C),
                    0xFC46 => array(0x645, 0x62D), 0xFC47 => array(0x645, 0x62E), 0xFC48 => array(0x645, 0x645),
                    0xFC49 => array(0x645, 0x649), 0xFC4A => array(0x645, 0x64A), 0xFC4B => array(0x646, 0x62C),
                    0xFC4C => array(0x646, 0x62D), 0xFC4D => array(0x646, 0x62E), 0xFC4E => array(0x646, 0x645),
                    0xFC4F => array(0x646, 0x649), 0xFC50 => array(0x646, 0x64A), 0xFC51 => array(0x647, 0x62C),
                    0xFC52 => array(0x647, 0x645), 0xFC53 => array(0x647, 0x649), 0xFC54 => array(0x647, 0x64A),
                    0xFC55 => array(0x64A, 0x62C), 0xFC56 => array(0x64A, 0x62D), 0xFC57 => array(0x64A, 0x62E),
                    0xFC58 => array(0x64A, 0x645), 0xFC59 => array(0x64A, 0x649), 0xFC5A => array(0x64A, 0x64A),
                    0xFC5B => array(0x630, 0x670), 0xFC5C => array(0x631, 0x670), 0xFC5D => array(0x649, 0x670),
                    0xFC64 => array(0x626, 0x631), 0xFC65 => array(0x626, 0x632), 0xFC66 => array(0x626, 0x645),
                    0xFC67 => array(0x626, 0x646), 0xFC68 => array(0x626, 0x649), 0xFC69 => array(0x626, 0x64A),
                    0xFC6A => array(0x628, 0x631), 0xFC6B => array(0x628, 0x632), 0xFC6C => array(0x628, 0x645),
                    0xFC6D => array(0x628, 0x646), 0xFC6E => array(0x628, 0x649), 0xFC6F => array(0x628, 0x64A),
                    0xFC70 => array(0x62A, 0x631), 0xFC71 => array(0x62A, 0x632), 0xFC72 => array(0x62A, 0x645),
                    0xFC73 => array(0x62A, 0x646), 0xFC74 => array(0x62A, 0x649), 0xFC75 => array(0x62A, 0x64A),
                    0xFC76 => array(0x62B, 0x631), 0xFC77 => array(0x62B, 0x632), 0xFC78 => array(0x62B, 0x645),
                    0xFC79 => array(0x62B, 0x646), 0xFC7A => array(0x62B, 0x649), 0xFC7B => array(0x62B, 0x64A),
                    0xFC7C => array(0x641, 0x649), 0xFC7D => array(0x641, 0x64A), 0xFC7E => array(0x642, 0x649),
                    0xFC7F => array(0x642, 0x64A), 0xFC80 => array(0x643, 0x627), 0xFC81 => array(0x643, 0x644),
                    0xFC82 => array(0x643, 0x645), 0xFC83 => array(0x643, 0x649), 0xFC84 => array(0x643, 0x64A),
                    0xFC85 => array(0x644, 0x645), 0xFC86 => array(0x644, 0x649), 0xFC87 => array(0x644, 0x64A),
                    0xFC88 => array(0x645, 0x627), 0xFC89 => array(0x645, 0x645), 0xFC8A => array(0x646, 0x631),
                    0xFC8B => array(0x646, 0x632), 0xFC8C => array(0x646, 0x645), 0xFC8D => array(0x646, 0x646),
                    0xFC8E => array(0x646, 0x649), 0xFC8F => array(0x646, 0x64A), 0xFC90 => array(0x649, 0x670),
                    0xFC91 => array(0x64A, 0x631), 0xFC92 => array(0x64A, 0x632), 0xFC93 => array(0x64A, 0x645),
                    0xFC94 => array(0x64A, 0x646), 0xFC95 => array(0x64A, 0x649), 0xFC96 => array(0x64A, 0x64A),
                    0xFC97 => array(0x626, 0x62C), 0xFC98 => array(0x626, 0x62D), 0xFC99 => array(0x626, 0x62E),
                    0xFC9A => array(0x626, 0x645), 0xFC9B => array(0x626, 0x647), 0xFC9C => array(0x628, 0x62C),
                    0xFC9D => array(0x628, 0x62D), 0xFC9E => array(0x628, 0x62E), 0xFC9F => array(0x628, 0x645),
                    0xFCA0 => array(0x628, 0x647), 0xFCA1 => array(0x62A, 0x62C), 0xFCA2 => array(0x62A, 0x62D),
                    0xFCA3 => array(0x62A, 0x62E), 0xFCA4 => array(0x62A, 0x645), 0xFCA5 => array(0x62A, 0x647),
                    0xFCA6 => array(0x62B, 0x645), 0xFCA7 => array(0x62C, 0x62D), 0xFCA8 => array(0x62C, 0x645),
                    0xFCA9 => array(0x62D, 0x62C), 0xFCAA => array(0x62D, 0x645), 0xFCAB => array(0x62E, 0x62C),
                    0xFCAC => array(0x62E, 0x645), 0xFCAD => array(0x633, 0x62C), 0xFCAE => array(0x633, 0x62D),
                    0xFCAF => array(0x633, 0x62E), 0xFCB0 => array(0x633, 0x645), 0xFCB1 => array(0x635, 0x62D),
                    0xFCB2 => array(0x635, 0x62E), 0xFCB3 => array(0x635, 0x645), 0xFCB4 => array(0x636, 0x62C),
                    0xFCB5 => array(0x636, 0x62D), 0xFCB6 => array(0x636, 0x62E), 0xFCB7 => array(0x636, 0x645),
                    0xFCB8 => array(0x637, 0x62D), 0xFCB9 => array(0x638, 0x645), 0xFCBA => array(0x639, 0x62C),
                    0xFCBB => array(0x639, 0x645), 0xFCBC => array(0x63A, 0x62C), 0xFCBD => array(0x63A, 0x645),
                    0xFCBE => array(0x641, 0x62C), 0xFCBF => array(0x641, 0x62D), 0xFCC0 => array(0x641, 0x62E),
                    0xFCC1 => array(0x641, 0x645), 0xFCC2 => array(0x642, 0x62D), 0xFCC3 => array(0x642, 0x645),
                    0xFCC4 => array(0x643, 0x62C), 0xFCC5 => array(0x643, 0x62D), 0xFCC6 => array(0x643, 0x62E),
                    0xFCC7 => array(0x643, 0x644), 0xFCC8 => array(0x643, 0x645), 0xFCC9 => array(0x644, 0x62C),
                    0xFCCA => array(0x644, 0x62D), 0xFCCB => array(0x644, 0x62E), 0xFCCC => array(0x644, 0x645),
                    0xFCCD => array(0x644, 0x647), 0xFCCE => array(0x645, 0x62C), 0xFCCF => array(0x645, 0x62D),
                    0xFCD0 => array(0x645, 0x62E), 0xFCD1 => array(0x645, 0x645), 0xFCD2 => array(0x646, 0x62C),
                    0xFCD3 => array(0x646, 0x62D), 0xFCD4 => array(0x646, 0x62E), 0xFCD5 => array(0x646, 0x645),
                    0xFCD6 => array(0x646, 0x647), 0xFCD7 => array(0x647, 0x62C), 0xFCD8 => array(0x647, 0x645),
                    0xFCD9 => array(0x647, 0x670), 0xFCDA => array(0x64A, 0x62C), 0xFCDB => array(0x64A, 0x62D),
                    0xFCDC => array(0x64A, 0x62E), 0xFCDD => array(0x64A, 0x645), 0xFCDE => array(0x64A, 0x647),
                    0xFCDF => array(0x626, 0x645), 0xFCE0 => array(0x626, 0x647), 0xFCE1 => array(0x628, 0x645),
                    0xFCE2 => array(0x628, 0x647), 0xFCE3 => array(0x62A, 0x645), 0xFCE4 => array(0x62A, 0x647),
                    0xFCE5 => array(0x62B, 0x645), 0xFCE6 => array(0x62B, 0x647), 0xFCE7 => array(0x633, 0x645),
                    0xFCE8 => array(0x633, 0x647), 0xFCE9 => array(0x634, 0x645), 0xFCEA => array(0x634, 0x647),
                    0xFCEB => array(0x643, 0x644), 0xFCEC => array(0x643, 0x645), 0xFCED => array(0x644, 0x645),
                    0xFCEE => array(0x646, 0x645), 0xFCEF => array(0x646, 0x647), 0xFCF0 => array(0x64A, 0x645),
                    0xFCF1 => array(0x64A, 0x647), 0xFCF2 => array(0x640, 0x64E, 0x651), 0xFCF3 => array(0x640, 0x64F, 0x651),
                    0xFCF4 => array(0x640, 0x650, 0x651), 0xFCF5 => array(0x637, 0x649), 0xFCF6 => array(0x637, 0x64A),
                    0xFCF7 => array(0x639, 0x649), 0xFCF8 => array(0x639, 0x64A), 0xFCF9 => array(0x63A, 0x649),
                    0xFCFA => array(0x63A, 0x64A), 0xFCFB => array(0x633, 0x649), 0xFCFC => array(0x633, 0x64A),
                    0xFCFD => array(0x634, 0x649), 0xFCFE => array(0x634, 0x64A), 0xFCFF => array(0x62D, 0x649),
                    0xFD00 => array(0x62D, 0x64A), 0xFD01 => array(0x62C, 0x649), 0xFD02 => array(0x62C, 0x64A),
                    0xFD03 => array(0x62E, 0x649), 0xFD04 => array(0x62E, 0x64A), 0xFD05 => array(0x635, 0x649),
                    0xFD06 => array(0x635, 0x64A), 0xFD07 => array(0x636, 0x649), 0xFD08 => array(0x636, 0x64A),
                    0xFD09 => array(0x634, 0x62C), 0xFD0A => array(0x634, 0x62D), 0xFD0B => array(0x634, 0x62E),
                    0xFD0C => array(0x634, 0x645), 0xFD0D => array(0x634, 0x631), 0xFD0E => array(0x633, 0x631),
                    0xFD0F => array(0x635, 0x631), 0xFD10 => array(0x636, 0x631), 0xFD11 => array(0x637, 0x649),
                    0xFD12 => array(0x637, 0x64A), 0xFD13 => array(0x639, 0x649), 0xFD14 => array(0x639, 0x64A),
                    0xFD15 => array(0x63A, 0x649), 0xFD16 => array(0x63A, 0x64A), 0xFD17 => array(0x633, 0x649),
                    0xFD18 => array(0x633, 0x64A), 0xFD19 => array(0x634, 0x649), 0xFD1A => array(0x634, 0x64A),
                    0xFD1B => array(0x62D, 0x649), 0xFD1C => array(0x62D, 0x64A), 0xFD1D => array(0x62C, 0x649),
                    0xFD1E => array(0x62C, 0x64A), 0xFD1F => array(0x62E, 0x649), 0xFD20 => array(0x62E, 0x64A),
                    0xFD21 => array(0x635, 0x649), 0xFD22 => array(0x635, 0x64A), 0xFD23 => array(0x636, 0x649),
                    0xFD24 => array(0x636, 0x64A), 0xFD25 => array(0x634, 0x62C), 0xFD26 => array(0x634, 0x62D),
                    0xFD27 => array(0x634, 0x62E), 0xFD28 => array(0x634, 0x645), 0xFD29 => array(0x634, 0x631),
                    0xFD2A => array(0x633, 0x631), 0xFD2B => array(0x635, 0x631), 0xFD2C => array(0x636, 0x631),
                    0xFD2D => array(0x634, 0x62C), 0xFD2E => array(0x634, 0x62D), 0xFD2F => array(0x634, 0x62E),
                    0xFD30 => array(0x634, 0x645), 0xFD31 => array(0x633, 0x647), 0xFD32 => array(0x634, 0x647),
                    0xFD33 => array(0x637, 0x645), 0xFD34 => array(0x633, 0x62C), 0xFD35 => array(0x633, 0x62D),
                    0xFD36 => array(0x633, 0x62E), 0xFD37 => array(0x634, 0x62C), 0xFD38 => array(0x634, 0x62D),
                    0xFD39 => array(0x634, 0x62E), 0xFD3A => array(0x637, 0x645), 0xFD3B => array(0x638, 0x645),
                    0xFD3C => array(0x627, 0x64B), 0xFD3D => array(0x627, 0x64B), 0xFD50 => array(0x62A, 0x62C, 0x645),
                    0xFD51 => array(0x62A, 0x62D, 0x62C), 0xFD52 => array(0x62A, 0x62D, 0x62C), 0xFD53 => array(0x62A, 0x62D, 0x645),
                    0xFD54 => array(0x62A, 0x62E, 0x645), 0xFD55 => array(0x62A, 0x645, 0x62C), 0xFD56 => array(0x62A, 0x645, 0x62D),
                    0xFD57 => array(0x62A, 0x645, 0x62E), 0xFD58 => array(0x62C, 0x645, 0x62D), 0xFD59 => array(0x62C, 0x645, 0x62D),
                    0xFD5A => array(0x62D, 0x645, 0x64A), 0xFD5B => array(0x62D, 0x645, 0x649), 0xFD5C => array(0x633, 0x62D, 0x62C),
                    0xFD5D => array(0x633, 0x62C, 0x62D), 0xFD5E => array(0x633, 0x62C, 0x649), 0xFD5F => array(0x633, 0x645, 0x62D),
                    0xFD60 => array(0x633, 0x645, 0x62D), 0xFD61 => array(0x633, 0x645, 0x62C), 0xFD62 => array(0x633, 0x645, 0x645),
                    0xFD63 => array(0x633, 0x645, 0x645), 0xFD64 => array(0x635, 0x62D, 0x62D), 0xFD65 => array(0x635, 0x62D, 0x62D),
                    0xFD66 => array(0x635, 0x645, 0x645), 0xFD67 => array(0x634, 0x62D, 0x645), 0xFD68 => array(0x634, 0x62D, 0x645),
                    0xFD69 => array(0x634, 0x62C, 0x64A), 0xFD6A => array(0x634, 0x645, 0x62E), 0xFD6B => array(0x634, 0x645, 0x62E),
                    0xFD6C => array(0x634, 0x645, 0x645), 0xFD6D => array(0x634, 0x645, 0x645), 0xFD6E => array(0x636, 0x62D, 0x649),
                    0xFD6F => array(0x636, 0x62E, 0x645), 0xFD70 => array(0x636, 0x62E, 0x645), 0xFD71 => array(0x637, 0x645, 0x62D),
                    0xFD72 => array(0x637, 0x645, 0x62D), 0xFD73 => array(0x637, 0x645, 0x645), 0xFD74 => array(0x637, 0x645, 0x64A),
                    0xFD75 => array(0x639, 0x62C, 0x645), 0xFD76 => array(0x639, 0x645, 0x645), 0xFD77 => array(0x639, 0x645, 0x645),
                    0xFD78 => array(0x639, 0x645, 0x649), 0xFD79 => array(0x63A, 0x645, 0x645), 0xFD7A => array(0x63A, 0x645, 0x64A),
                    0xFD7B => array(0x63A, 0x645, 0x649), 0xFD7C => array(0x641, 0x62E, 0x645), 0xFD7D => array(0x641, 0x62E, 0x645),
                    0xFD7E => array(0x642, 0x645, 0x62D), 0xFD7F => array(0x642, 0x645, 0x645), 0xFD80 => array(0x644, 0x62D, 0x645),
                    0xFD81 => array(0x644, 0x62D, 0x64A), 0xFD82 => array(0x644, 0x62D, 0x649), 0xFD83 => array(0x644, 0x62C, 0x62C),
                    0xFD84 => array(0x644, 0x62C, 0x62C), 0xFD85 => array(0x644, 0x62E, 0x645), 0xFD86 => array(0x644, 0x62E, 0x645),
                    0xFD87 => array(0x644, 0x645, 0x62D), 0xFD88 => array(0x644, 0x645, 0x62D), 0xFD89 => array(0x645, 0x62D, 0x62C),
                    0xFD8A => array(0x645, 0x62D, 0x645), 0xFD8B => array(0x645, 0x62D, 0x64A), 0xFD8C => array(0x645, 0x62C, 0x62D),
                    0xFD8D => array(0x645, 0x62C, 0x645), 0xFD8E => array(0x645, 0x62E, 0x62C), 0xFD8F => array(0x645, 0x62E, 0x645),
                    0xFD92 => array(0x645, 0x62C, 0x62E), 0xFD93 => array(0x647, 0x645, 0x62C), 0xFD94 => array(0x647, 0x645, 0x645),
                    0xFD95 => array(0x646, 0x62D, 0x645), 0xFD96 => array(0x646, 0x62D, 0x649), 0xFD97 => array(0x646, 0x62C, 0x645),
                    0xFD98 => array(0x646, 0x62C, 0x645), 0xFD99 => array(0x646, 0x62C, 0x649), 0xFD9A => array(0x646, 0x645, 0x64A),
                    0xFD9B => array(0x646, 0x645, 0x649), 0xFD9C => array(0x64A, 0x645, 0x645), 0xFD9D => array(0x64A, 0x645, 0x645),
                    0xFD9E => array(0x628, 0x62E, 0x64A), 0xFD9F => array(0x62A, 0x62C, 0x64A), 0xFDA0 => array(0x62A, 0x62C, 0x649),
                    0xFDA1 => array(0x62A, 0x62E, 0x64A), 0xFDA2 => array(0x62A, 0x62E, 0x649), 0xFDA3 => array(0x62A, 0x645, 0x64A),
                    0xFDA4 => array(0x62A, 0x645, 0x649), 0xFDA5 => array(0x62C, 0x645, 0x64A), 0xFDA6 => array(0x62C, 0x62D, 0x649),
                    0xFDA7 => array(0x62C, 0x645, 0x649), 0xFDA8 => array(0x633, 0x62E, 0x649), 0xFDA9 => array(0x635, 0x62D, 0x64A),
                    0xFDAA => array(0x634, 0x62D, 0x64A), 0xFDAB => array(0x636, 0x62D, 0x64A), 0xFDAC => array(0x644, 0x62C, 0x64A),
                    0xFDAD => array(0x644, 0x645, 0x64A), 0xFDAE => array(0x64A, 0x62D, 0x64A), 0xFDAF => array(0x64A, 0x62C, 0x64A),
                    0xFDB0 => array(0x64A, 0x645, 0x64A), 0xFDB1 => array(0x645, 0x645, 0x64A), 0xFDB2 => array(0x642, 0x645, 0x64A),
                    0xFDB3 => array(0x646, 0x62D, 0x64A), 0xFDB4 => array(0x642, 0x645, 0x62D), 0xFDB5 => array(0x644, 0x62D, 0x645),
                    0xFDB6 => array(0x639, 0x645, 0x64A), 0xFDB7 => array(0x643, 0x645, 0x64A), 0xFDB8 => array(0x646, 0x62C, 0x62D),
                    0xFDB9 => array(0x645, 0x62E, 0x64A), 0xFDBA => array(0x644, 0x62C, 0x645), 0xFDBB => array(0x643, 0x645, 0x645),
                    0xFDBC => array(0x644, 0x62C, 0x645), 0xFDBD => array(0x646, 0x62C, 0x62D), 0xFDBE => array(0x62C, 0x62D, 0x64A),
                    0xFDBF => array(0x62D, 0x62C, 0x64A), 0xFDC0 => array(0x645, 0x62C, 0x64A), 0xFDC1 => array(0x641, 0x645, 0x64A),
                    0xFDC2 => array(0x628, 0x62D, 0x64A), 0xFDC3 => array(0x643, 0x645, 0x645), 0xFDC4 => array(0x639, 0x62C, 0x645),
                    0xFDC5 => array(0x635, 0x645, 0x645), 0xFDC6 => array(0x633, 0x62E, 0x64A), 0xFDC7 => array(0x646, 0x62C, 0x64A),
                    0xFDF0 => array(0x635, 0x644, 0x6D2), 0xFDF1 => array(0x642, 0x644, 0x6D2), 0xFDF2 => array(0x627, 0x644, 0x644, 0x647),
                    0xFDF3 => array(0x627, 0x643, 0x628, 0x631), 0xFDF4 => array(0x645, 0x62D, 0x645, 0x62F), 0xFDF5 => array(0x635, 0x644, 0x639, 0x645),
                    0xFDF6 => array(0x631, 0x633, 0x648, 0x644), 0xFDF7 => array(0x639, 0x644, 0x64A, 0x647), 0xFDF8 => array(0x648, 0x633, 0x644, 0x645),
                    0xFDF9 => array(0x635, 0x644, 0x649), 0xFDFC => array(0x631, 0x6CC, 0x627, 0x644), 0xFE11 => array(0x3001),
                    0xFE17 => array(0x3016), 0xFE18 => array(0x3017), 0xFE31 => array(0x2014),
                    0xFE32 => array(0x2013), 0xFE39 => array(0x3014), 0xFE3A => array(0x3015),
                    0xFE3B => array(0x3010), 0xFE3C => array(0x3011), 0xFE3D => array(0x300A),
                    0xFE3E => array(0x300B), 0xFE3F => array(0x3008), 0xFE40 => array(0x3009),
                    0xFE41 => array(0x300C), 0xFE42 => array(0x300D), 0xFE43 => array(0x300E),
                    0xFE44 => array(0x300F), 0xFE51 => array(0x3001), 0xFE58 => array(0x2014),
                    0xFE5D => array(0x3014), 0xFE5E => array(0x3015), 0xFE63 => array(0x2D),
                    0xFE71 => array(0x640, 0x64B), 0xFE77 => array(0x640, 0x64E), 0xFE79 => array(0x640, 0x64F),
                    0xFE7B => array(0x640, 0x650), 0xFE7D => array(0x640, 0x651), 0xFE7F => array(0x640, 0x652),
                    0xFE80 => array(0x621), 0xFE81 => array(0x622), 0xFE82 => array(0x622),
                    0xFE83 => array(0x623), 0xFE84 => array(0x623), 0xFE85 => array(0x624),
                    0xFE86 => array(0x624), 0xFE87 => array(0x625), 0xFE88 => array(0x625),
                    0xFE89 => array(0x626), 0xFE8A => array(0x626), 0xFE8B => array(0x626),
                    0xFE8C => array(0x626), 0xFE8D => array(0x627), 0xFE8E => array(0x627),
                    0xFE8F => array(0x628), 0xFE90 => array(0x628), 0xFE91 => array(0x628),
                    0xFE92 => array(0x628), 0xFE93 => array(0x629), 0xFE94 => array(0x629),
                    0xFE95 => array(0x62A), 0xFE96 => array(0x62A), 0xFE97 => array(0x62A),
                    0xFE98 => array(0x62A), 0xFE99 => array(0x62B), 0xFE9A => array(0x62B),
                    0xFE9B => array(0x62B), 0xFE9C => array(0x62B), 0xFE9D => array(0x62C),
                    0xFE9E => array(0x62C), 0xFE9F => array(0x62C), 0xFEA0 => array(0x62C),
                    0xFEA1 => array(0x62D), 0xFEA2 => array(0x62D), 0xFEA3 => array(0x62D),
                    0xFEA4 => array(0x62D), 0xFEA5 => array(0x62E), 0xFEA6 => array(0x62E),
                    0xFEA7 => array(0x62E), 0xFEA8 => array(0x62E), 0xFEA9 => array(0x62F),
                    0xFEAA => array(0x62F), 0xFEAB => array(0x630), 0xFEAC => array(0x630),
                    0xFEAD => array(0x631), 0xFEAE => array(0x631), 0xFEAF => array(0x632),
                    0xFEB0 => array(0x632), 0xFEB1 => array(0x633), 0xFEB2 => array(0x633),
                    0xFEB3 => array(0x633), 0xFEB4 => array(0x633), 0xFEB5 => array(0x634),
                    0xFEB6 => array(0x634), 0xFEB7 => array(0x634), 0xFEB8 => array(0x634),
                    0xFEB9 => array(0x635), 0xFEBA => array(0x635), 0xFEBB => array(0x635),
                    0xFEBC => array(0x635), 0xFEBD => array(0x636), 0xFEBE => array(0x636),
                    0xFEBF => array(0x636), 0xFEC0 => array(0x636), 0xFEC1 => array(0x637),
                    0xFEC2 => array(0x637), 0xFEC3 => array(0x637), 0xFEC4 => array(0x637),
                    0xFEC5 => array(0x638), 0xFEC6 => array(0x638), 0xFEC7 => array(0x638),
                    0xFEC8 => array(0x638), 0xFEC9 => array(0x639), 0xFECA => array(0x639),
                    0xFECB => array(0x639), 0xFECC => array(0x639), 0xFECD => array(0x63A),
                    0xFECE => array(0x63A), 0xFECF => array(0x63A), 0xFED0 => array(0x63A),
                    0xFED1 => array(0x641), 0xFED2 => array(0x641), 0xFED3 => array(0x641),
                    0xFED4 => array(0x641), 0xFED5 => array(0x642), 0xFED6 => array(0x642),
                    0xFED7 => array(0x642), 0xFED8 => array(0x642), 0xFED9 => array(0x643),
                    0xFEDA => array(0x643), 0xFEDB => array(0x643), 0xFEDC => array(0x643),
                    0xFEDD => array(0x644), 0xFEDE => array(0x644), 0xFEDF => array(0x644),
                    0xFEE0 => array(0x644), 0xFEE1 => array(0x645), 0xFEE2 => array(0x645),
                    0xFEE3 => array(0x645), 0xFEE4 => array(0x645), 0xFEE5 => array(0x646),
                    0xFEE6 => array(0x646), 0xFEE7 => array(0x646), 0xFEE8 => array(0x646),
                    0xFEE9 => array(0x647), 0xFEEA => array(0x647), 0xFEEB => array(0x647),
                    0xFEEC => array(0x647), 0xFEED => array(0x648), 0xFEEE => array(0x648),
                    0xFEEF => array(0x649), 0xFEF0 => array(0x649), 0xFEF1 => array(0x64A),
                    0xFEF2 => array(0x64A), 0xFEF3 => array(0x64A), 0xFEF4 => array(0x64A),
                    0xFEF5 => array(0x644, 0x622), 0xFEF6 => array(0x644, 0x622), 0xFEF7 => array(0x644, 0x623),
                    0xFEF8 => array(0x644, 0x623), 0xFEF9 => array(0x644, 0x625), 0xFEFA => array(0x644, 0x625),
                    0xFEFB => array(0x644, 0x627), 0xFEFC => array(0x644, 0x627), 0xFF0D => array(0x2D),
                    0xFF0E => array(0x2E), 0xFF10 => array(0x30), 0xFF11 => array(0x31),
                    0xFF12 => array(0x32), 0xFF13 => array(0x33), 0xFF14 => array(0x34),
                    0xFF15 => array(0x35), 0xFF16 => array(0x36), 0xFF17 => array(0x37),
                    0xFF18 => array(0x38), 0xFF19 => array(0x39), 0xFF21 => array(0x61),
                    0xFF22 => array(0x62), 0xFF23 => array(0x63), 0xFF24 => array(0x64),
                    0xFF25 => array(0x65), 0xFF26 => array(0x66), 0xFF27 => array(0x67),
                    0xFF28 => array(0x68), 0xFF29 => array(0x69), 0xFF2A => array(0x6A),
                    0xFF2B => array(0x6B), 0xFF2C => array(0x6C), 0xFF2D => array(0x6D),
                    0xFF2E => array(0x6E), 0xFF2F => array(0x6F), 0xFF30 => array(0x70),
                    0xFF31 => array(0x71), 0xFF32 => array(0x72), 0xFF33 => array(0x73),
                    0xFF34 => array(0x74), 0xFF35 => array(0x75), 0xFF36 => array(0x76),
                    0xFF37 => array(0x77), 0xFF38 => array(0x78), 0xFF39 => array(0x79),
                    0xFF3A => array(0x7A), 0xFF41 => array(0x61), 0xFF42 => array(0x62),
                    0xFF43 => array(0x63), 0xFF44 => array(0x64), 0xFF45 => array(0x65),
                    0xFF46 => array(0x66), 0xFF47 => array(0x67), 0xFF48 => array(0x68),
                    0xFF49 => array(0x69), 0xFF4A => array(0x6A), 0xFF4B => array(0x6B),
                    0xFF4C => array(0x6C), 0xFF4D => array(0x6D), 0xFF4E => array(0x6E),
                    0xFF4F => array(0x6F), 0xFF50 => array(0x70), 0xFF51 => array(0x71),
                    0xFF52 => array(0x72), 0xFF53 => array(0x73), 0xFF54 => array(0x74),
                    0xFF55 => array(0x75), 0xFF56 => array(0x76), 0xFF57 => array(0x77),
                    0xFF58 => array(0x78), 0xFF59 => array(0x79), 0xFF5A => array(0x7A),
                    0xFF5F => array(0x2985), 0xFF60 => array(0x2986), 0xFF61 => array(0x2E),
                    0xFF62 => array(0x300C), 0xFF63 => array(0x300D), 0xFF64 => array(0x3001),
                    0xFF65 => array(0x30FB), 0xFF66 => array(0x30F2), 0xFF67 => array(0x30A1),
                    0xFF68 => array(0x30A3), 0xFF69 => array(0x30A5), 0xFF6A => array(0x30A7),
                    0xFF6B => array(0x30A9), 0xFF6C => array(0x30E3), 0xFF6D => array(0x30E5),
                    0xFF6E => array(0x30E7), 0xFF6F => array(0x30C3), 0xFF70 => array(0x30FC),
                    0xFF71 => array(0x30A2), 0xFF72 => array(0x30A4), 0xFF73 => array(0x30A6),
                    0xFF74 => array(0x30A8), 0xFF75 => array(0x30AA), 0xFF76 => array(0x30AB),
                    0xFF77 => array(0x30AD), 0xFF78 => array(0x30AF), 0xFF79 => array(0x30B1),
                    0xFF7A => array(0x30B3), 0xFF7B => array(0x30B5), 0xFF7C => array(0x30B7),
                    0xFF7D => array(0x30B9), 0xFF7E => array(0x30BB), 0xFF7F => array(0x30BD),
                    0xFF80 => array(0x30BF), 0xFF81 => array(0x30C1), 0xFF82 => array(0x30C4),
                    0xFF83 => array(0x30C6), 0xFF84 => array(0x30C8), 0xFF85 => array(0x30CA),
                    0xFF86 => array(0x30CB), 0xFF87 => array(0x30CC), 0xFF88 => array(0x30CD),
                    0xFF89 => array(0x30CE), 0xFF8A => array(0x30CF), 0xFF8B => array(0x30D2),
                    0xFF8C => array(0x30D5), 0xFF8D => array(0x30D8), 0xFF8E => array(0x30DB),
                    0xFF8F => array(0x30DE), 0xFF90 => array(0x30DF), 0xFF91 => array(0x30E0),
                    0xFF92 => array(0x30E1), 0xFF93 => array(0x30E2), 0xFF94 => array(0x30E4),
                    0xFF95 => array(0x30E6), 0xFF96 => array(0x30E8), 0xFF97 => array(0x30E9),
                    0xFF98 => array(0x30EA), 0xFF99 => array(0x30EB), 0xFF9A => array(0x30EC),
                    0xFF9B => array(0x30ED), 0xFF9C => array(0x30EF), 0xFF9D => array(0x30F3),
                    0xFF9E => array(0x3099), 0xFF9F => array(0x309A), 0xFFA1 => array(0x1100),
                    0xFFA2 => array(0x1101), 0xFFA3 => array(0x11AA), 0xFFA4 => array(0x1102),
                    0xFFA5 => array(0x11AC), 0xFFA6 => array(0x11AD), 0xFFA7 => array(0x1103),
                    0xFFA8 => array(0x1104), 0xFFA9 => array(0x1105), 0xFFAA => array(0x11B0),
                    0xFFAB => array(0x11B1), 0xFFAC => array(0x11B2), 0xFFAD => array(0x11B3),
                    0xFFAE => array(0x11B4), 0xFFAF => array(0x11B5), 0xFFB0 => array(0x111A),
                    0xFFB1 => array(0x1106), 0xFFB2 => array(0x1107), 0xFFB3 => array(0x1108),
                    0xFFB4 => array(0x1121), 0xFFB5 => array(0x1109), 0xFFB6 => array(0x110A),
                    0xFFB7 => array(0x110B), 0xFFB8 => array(0x110C), 0xFFB9 => array(0x110D),
                    0xFFBA => array(0x110E), 0xFFBB => array(0x110F), 0xFFBC => array(0x1110),
                    0xFFBD => array(0x1111), 0xFFBE => array(0x1112), 0xFFC2 => array(0x1161),
                    0xFFC3 => array(0x1162), 0xFFC4 => array(0x1163), 0xFFC5 => array(0x1164),
                    0xFFC6 => array(0x1165), 0xFFC7 => array(0x1166), 0xFFCA => array(0x1167),
                    0xFFCB => array(0x1168), 0xFFCC => array(0x1169), 0xFFCD => array(0x116A),
                    0xFFCE => array(0x116B), 0xFFCF => array(0x116C), 0xFFD2 => array(0x116D),
                    0xFFD3 => array(0x116E), 0xFFD4 => array(0x116F), 0xFFD5 => array(0x1170),
                    0xFFD6 => array(0x1171), 0xFFD7 => array(0x1172), 0xFFDA => array(0x1173),
                    0xFFDB => array(0x1174), 0xFFDC => array(0x1175), 0xFFE0 => array(0xA2),
                    0xFFE1 => array(0xA3), 0xFFE2 => array(0xAC), 0xFFE4 => array(0xA6),
                    0xFFE5 => array(0xA5), 0xFFE6 => array(0x20A9), 0xFFE8 => array(0x2502),
                    0xFFE9 => array(0x2190), 0xFFEA => array(0x2191), 0xFFEB => array(0x2192),
                    0xFFEC => array(0x2193), 0xFFED => array(0x25A0), 0xFFEE => array(0x25CB),
                    0x10400 => array(0x10428), 0x10401 => array(0x10429), 0x10402 => array(0x1042A),
                    0x10403 => array(0x1042B), 0x10404 => array(0x1042C), 0x10405 => array(0x1042D),
                    0x10406 => array(0x1042E), 0x10407 => array(0x1042F), 0x10408 => array(0x10430),
                    0x10409 => array(0x10431), 0x1040A => array(0x10432), 0x1040B => array(0x10433),
                    0x1040C => array(0x10434), 0x1040D => array(0x10435), 0x1040E => array(0x10436),
                    0x1040F => array(0x10437), 0x10410 => array(0x10438), 0x10411 => array(0x10439),
                    0x10412 => array(0x1043A), 0x10413 => array(0x1043B), 0x10414 => array(0x1043C),
                    0x10415 => array(0x1043D), 0x10416 => array(0x1043E), 0x10417 => array(0x1043F),
                    0x10418 => array(0x10440), 0x10419 => array(0x10441), 0x1041A => array(0x10442),
                    0x1041B => array(0x10443), 0x1041C => array(0x10444), 0x1041D => array(0x10445),
                    0x1041E => array(0x10446), 0x1041F => array(0x10447), 0x10420 => array(0x10448),
                    0x10421 => array(0x10449), 0x10422 => array(0x1044A), 0x10423 => array(0x1044B),
                    0x10424 => array(0x1044C), 0x10425 => array(0x1044D), 0x10426 => array(0x1044E),
                    0x10427 => array(0x1044F), 0x118A0 => array(0x118C0), 0x118A1 => array(0x118C1),
                    0x118A2 => array(0x118C2), 0x118A3 => array(0x118C3), 0x118A4 => array(0x118C4),
                    0x118A5 => array(0x118C5), 0x118A6 => array(0x118C6), 0x118A7 => array(0x118C7),
                    0x118A8 => array(0x118C8), 0x118A9 => array(0x118C9), 0x118AA => array(0x118CA),
                    0x118AB => array(0x118CB), 0x118AC => array(0x118CC), 0x118AD => array(0x118CD),
                    0x118AE => array(0x118CE), 0x118AF => array(0x118CF), 0x118B0 => array(0x118D0),
                    0x118B1 => array(0x118D1), 0x118B2 => array(0x118D2), 0x118B3 => array(0x118D3),
                    0x118B4 => array(0x118D4), 0x118B5 => array(0x118D5), 0x118B6 => array(0x118D6),
                    0x118B7 => array(0x118D7), 0x118B8 => array(0x118D8), 0x118B9 => array(0x118D9),
                    0x118BA => array(0x118DA), 0x118BB => array(0x118DB), 0x118BC => array(0x118DC),
                    0x118BD => array(0x118DD), 0x118BE => array(0x118DE), 0x118BF => array(0x118DF),
                    0x1D15E => array(0x1D157, 0x1D165), 0x1D15F => array(0x1D158, 0x1D165), 0x1D160 => array(0x1D158, 0x1D165, 0x1D16E),
                    0x1D161 => array(0x1D158, 0x1D165, 0x1D16F), 0x1D162 => array(0x1D158, 0x1D165, 0x1D170), 0x1D163 => array(0x1D158, 0x1D165, 0x1D171),
                    0x1D164 => array(0x1D158, 0x1D165, 0x1D172), 0x1D1BB => array(0x1D1B9, 0x1D165), 0x1D1BC => array(0x1D1BA, 0x1D165),
                    0x1D1BD => array(0x1D1B9, 0x1D165, 0x1D16E), 0x1D1BE => array(0x1D1BA, 0x1D165, 0x1D16E), 0x1D1BF => array(0x1D1B9, 0x1D165, 0x1D16F),
                    0x1D1C0 => array(0x1D1BA, 0x1D165, 0x1D16F), 0x1D400 => array(0x61), 0x1D401 => array(0x62),
                    0x1D402 => array(0x63), 0x1D403 => array(0x64), 0x1D404 => array(0x65),
                    0x1D405 => array(0x66), 0x1D406 => array(0x67), 0x1D407 => array(0x68),
                    0x1D408 => array(0x69), 0x1D409 => array(0x6A), 0x1D40A => array(0x6B),
                    0x1D40B => array(0x6C), 0x1D40C => array(0x6D), 0x1D40D => array(0x6E),
                    0x1D40E => array(0x6F), 0x1D40F => array(0x70), 0x1D410 => array(0x71),
                    0x1D411 => array(0x72), 0x1D412 => array(0x73), 0x1D413 => array(0x74),
                    0x1D414 => array(0x75), 0x1D415 => array(0x76), 0x1D416 => array(0x77),
                    0x1D417 => array(0x78), 0x1D418 => array(0x79), 0x1D419 => array(0x7A),
                    0x1D41A => array(0x61), 0x1D41B => array(0x62), 0x1D41C => array(0x63),
                    0x1D41D => array(0x64), 0x1D41E => array(0x65), 0x1D41F => array(0x66),
                    0x1D420 => array(0x67), 0x1D421 => array(0x68), 0x1D422 => array(0x69),
                    0x1D423 => array(0x6A), 0x1D424 => array(0x6B), 0x1D425 => array(0x6C),
                    0x1D426 => array(0x6D), 0x1D427 => array(0x6E), 0x1D428 => array(0x6F),
                    0x1D429 => array(0x70), 0x1D42A => array(0x71), 0x1D42B => array(0x72),
                    0x1D42C => array(0x73), 0x1D42D => array(0x74), 0x1D42E => array(0x75),
                    0x1D42F => array(0x76), 0x1D430 => array(0x77), 0x1D431 => array(0x78),
                    0x1D432 => array(0x79), 0x1D433 => array(0x7A), 0x1D434 => array(0x61),
                    0x1D435 => array(0x62), 0x1D436 => array(0x63), 0x1D437 => array(0x64),
                    0x1D438 => array(0x65), 0x1D439 => array(0x66), 0x1D43A => array(0x67),
                    0x1D43B => array(0x68), 0x1D43C => array(0x69), 0x1D43D => array(0x6A),
                    0x1D43E => array(0x6B), 0x1D43F => array(0x6C), 0x1D440 => array(0x6D),
                    0x1D441 => array(0x6E), 0x1D442 => array(0x6F), 0x1D443 => array(0x70),
                    0x1D444 => array(0x71), 0x1D445 => array(0x72), 0x1D446 => array(0x73),
                    0x1D447 => array(0x74), 0x1D448 => array(0x75), 0x1D449 => array(0x76),
                    0x1D44A => array(0x77), 0x1D44B => array(0x78), 0x1D44C => array(0x79),
                    0x1D44D => array(0x7A), 0x1D44E => array(0x61), 0x1D44F => array(0x62),
                    0x1D450 => array(0x63), 0x1D451 => array(0x64), 0x1D452 => array(0x65),
                    0x1D453 => array(0x66), 0x1D454 => array(0x67), 0x1D456 => array(0x69),
                    0x1D457 => array(0x6A), 0x1D458 => array(0x6B), 0x1D459 => array(0x6C),
                    0x1D45A => array(0x6D), 0x1D45B => array(0x6E), 0x1D45C => array(0x6F),
                    0x1D45D => array(0x70), 0x1D45E => array(0x71), 0x1D45F => array(0x72),
                    0x1D460 => array(0x73), 0x1D461 => array(0x74), 0x1D462 => array(0x75),
                    0x1D463 => array(0x76), 0x1D464 => array(0x77), 0x1D465 => array(0x78),
                    0x1D466 => array(0x79), 0x1D467 => array(0x7A), 0x1D468 => array(0x61),
                    0x1D469 => array(0x62), 0x1D46A => array(0x63), 0x1D46B => array(0x64),
                    0x1D46C => array(0x65), 0x1D46D => array(0x66), 0x1D46E => array(0x67),
                    0x1D46F => array(0x68), 0x1D470 => array(0x69), 0x1D471 => array(0x6A),
                    0x1D472 => array(0x6B), 0x1D473 => array(0x6C), 0x1D474 => array(0x6D),
                    0x1D475 => array(0x6E), 0x1D476 => array(0x6F), 0x1D477 => array(0x70),
                    0x1D478 => array(0x71), 0x1D479 => array(0x72), 0x1D47A => array(0x73),
                    0x1D47B => array(0x74), 0x1D47C => array(0x75), 0x1D47D => array(0x76),
                    0x1D47E => array(0x77), 0x1D47F => array(0x78), 0x1D480 => array(0x79),
                    0x1D481 => array(0x7A), 0x1D482 => array(0x61), 0x1D483 => array(0x62),
                    0x1D484 => array(0x63), 0x1D485 => array(0x64), 0x1D486 => array(0x65),
                    0x1D487 => array(0x66), 0x1D488 => array(0x67), 0x1D489 => array(0x68),
                    0x1D48A => array(0x69), 0x1D48B => array(0x6A), 0x1D48C => array(0x6B),
                    0x1D48D => array(0x6C), 0x1D48E => array(0x6D), 0x1D48F => array(0x6E),
                    0x1D490 => array(0x6F), 0x1D491 => array(0x70), 0x1D492 => array(0x71),
                    0x1D493 => array(0x72), 0x1D494 => array(0x73), 0x1D495 => array(0x74),
                    0x1D496 => array(0x75), 0x1D497 => array(0x76), 0x1D498 => array(0x77),
                    0x1D499 => array(0x78), 0x1D49A => array(0x79), 0x1D49B => array(0x7A),
                    0x1D49C => array(0x61), 0x1D49E => array(0x63), 0x1D49F => array(0x64),
                    0x1D4A2 => array(0x67), 0x1D4A5 => array(0x6A), 0x1D4A6 => array(0x6B),
                    0x1D4A9 => array(0x6E), 0x1D4AA => array(0x6F), 0x1D4AB => array(0x70),
                    0x1D4AC => array(0x71), 0x1D4AE => array(0x73), 0x1D4AF => array(0x74),
                    0x1D4B0 => array(0x75), 0x1D4B1 => array(0x76), 0x1D4B2 => array(0x77),
                    0x1D4B3 => array(0x78), 0x1D4B4 => array(0x79), 0x1D4B5 => array(0x7A),
                    0x1D4B6 => array(0x61), 0x1D4B7 => array(0x62), 0x1D4B8 => array(0x63),
                    0x1D4B9 => array(0x64), 0x1D4BB => array(0x66), 0x1D4BD => array(0x68),
                    0x1D4BE => array(0x69), 0x1D4BF => array(0x6A), 0x1D4C0 => array(0x6B),
                    0x1D4C1 => array(0x6C), 0x1D4C2 => array(0x6D), 0x1D4C3 => array(0x6E),
                    0x1D4C5 => array(0x70), 0x1D4C6 => array(0x71), 0x1D4C7 => array(0x72),
                    0x1D4C8 => array(0x73), 0x1D4C9 => array(0x74), 0x1D4CA => array(0x75),
                    0x1D4CB => array(0x76), 0x1D4CC => array(0x77), 0x1D4CD => array(0x78),
                    0x1D4CE => array(0x79), 0x1D4CF => array(0x7A), 0x1D4D0 => array(0x61),
                    0x1D4D1 => array(0x62), 0x1D4D2 => array(0x63), 0x1D4D3 => array(0x64),
                    0x1D4D4 => array(0x65), 0x1D4D5 => array(0x66), 0x1D4D6 => array(0x67),
                    0x1D4D7 => array(0x68), 0x1D4D8 => array(0x69), 0x1D4D9 => array(0x6A),
                    0x1D4DA => array(0x6B), 0x1D4DB => array(0x6C), 0x1D4DC => array(0x6D),
                    0x1D4DD => array(0x6E), 0x1D4DE => array(0x6F), 0x1D4DF => array(0x70),
                    0x1D4E0 => array(0x71), 0x1D4E1 => array(0x72), 0x1D4E2 => array(0x73),
                    0x1D4E3 => array(0x74), 0x1D4E4 => array(0x75), 0x1D4E5 => array(0x76),
                    0x1D4E6 => array(0x77), 0x1D4E7 => array(0x78), 0x1D4E8 => array(0x79),
                    0x1D4E9 => array(0x7A), 0x1D4EA => array(0x61), 0x1D4EB => array(0x62),
                    0x1D4EC => array(0x63), 0x1D4ED => array(0x64), 0x1D4EE => array(0x65),
                    0x1D4EF => array(0x66), 0x1D4F0 => array(0x67), 0x1D4F1 => array(0x68),
                    0x1D4F2 => array(0x69), 0x1D4F3 => array(0x6A), 0x1D4F4 => array(0x6B),
                    0x1D4F5 => array(0x6C), 0x1D4F6 => array(0x6D), 0x1D4F7 => array(0x6E),
                    0x1D4F8 => array(0x6F), 0x1D4F9 => array(0x70), 0x1D4FA => array(0x71),
                    0x1D4FB => array(0x72), 0x1D4FC => array(0x73), 0x1D4FD => array(0x74),
                    0x1D4FE => array(0x75), 0x1D4FF => array(0x76), 0x1D500 => array(0x77),
                    0x1D501 => array(0x78), 0x1D502 => array(0x79), 0x1D503 => array(0x7A),
                    0x1D504 => array(0x61), 0x1D505 => array(0x62), 0x1D507 => array(0x64),
                    0x1D508 => array(0x65), 0x1D509 => array(0x66), 0x1D50A => array(0x67),
                    0x1D50D => array(0x6A), 0x1D50E => array(0x6B), 0x1D50F => array(0x6C),
                    0x1D510 => array(0x6D), 0x1D511 => array(0x6E), 0x1D512 => array(0x6F),
                    0x1D513 => array(0x70), 0x1D514 => array(0x71), 0x1D516 => array(0x73),
                    0x1D517 => array(0x74), 0x1D518 => array(0x75), 0x1D519 => array(0x76),
                    0x1D51A => array(0x77), 0x1D51B => array(0x78), 0x1D51C => array(0x79),
                    0x1D51E => array(0x61), 0x1D51F => array(0x62), 0x1D520 => array(0x63),
                    0x1D521 => array(0x64), 0x1D522 => array(0x65), 0x1D523 => array(0x66),
                    0x1D524 => array(0x67), 0x1D525 => array(0x68), 0x1D526 => array(0x69),
                    0x1D527 => array(0x6A), 0x1D528 => array(0x6B), 0x1D529 => array(0x6C),
                    0x1D52A => array(0x6D), 0x1D52B => array(0x6E), 0x1D52C => array(0x6F),
                    0x1D52D => array(0x70), 0x1D52E => array(0x71), 0x1D52F => array(0x72),
                    0x1D530 => array(0x73), 0x1D531 => array(0x74), 0x1D532 => array(0x75),
                    0x1D533 => array(0x76), 0x1D534 => array(0x77), 0x1D535 => array(0x78),
                    0x1D536 => array(0x79), 0x1D537 => array(0x7A), 0x1D538 => array(0x61),
                    0x1D539 => array(0x62), 0x1D53B => array(0x64), 0x1D53C => array(0x65),
                    0x1D53D => array(0x66), 0x1D53E => array(0x67), 0x1D540 => array(0x69),
                    0x1D541 => array(0x6A), 0x1D542 => array(0x6B), 0x1D543 => array(0x6C),
                    0x1D544 => array(0x6D), 0x1D546 => array(0x6F), 0x1D54A => array(0x73),
                    0x1D54B => array(0x74), 0x1D54C => array(0x75), 0x1D54D => array(0x76),
                    0x1D54E => array(0x77), 0x1D54F => array(0x78), 0x1D550 => array(0x79),
                    0x1D552 => array(0x61), 0x1D553 => array(0x62), 0x1D554 => array(0x63),
                    0x1D555 => array(0x64), 0x1D556 => array(0x65), 0x1D557 => array(0x66),
                    0x1D558 => array(0x67), 0x1D559 => array(0x68), 0x1D55A => array(0x69),
                    0x1D55B => array(0x6A), 0x1D55C => array(0x6B), 0x1D55D => array(0x6C),
                    0x1D55E => array(0x6D), 0x1D55F => array(0x6E), 0x1D560 => array(0x6F),
                    0x1D561 => array(0x70), 0x1D562 => array(0x71), 0x1D563 => array(0x72),
                    0x1D564 => array(0x73), 0x1D565 => array(0x74), 0x1D566 => array(0x75),
                    0x1D567 => array(0x76), 0x1D568 => array(0x77), 0x1D569 => array(0x78),
                    0x1D56A => array(0x79), 0x1D56B => array(0x7A), 0x1D56C => array(0x61),
                    0x1D56D => array(0x62), 0x1D56E => array(0x63), 0x1D56F => array(0x64),
                    0x1D570 => array(0x65), 0x1D571 => array(0x66), 0x1D572 => array(0x67),
                    0x1D573 => array(0x68), 0x1D574 => array(0x69), 0x1D575 => array(0x6A),
                    0x1D576 => array(0x6B), 0x1D577 => array(0x6C), 0x1D578 => array(0x6D),
                    0x1D579 => array(0x6E), 0x1D57A => array(0x6F), 0x1D57B => array(0x70),
                    0x1D57C => array(0x71), 0x1D57D => array(0x72), 0x1D57E => array(0x73),
                    0x1D57F => array(0x74), 0x1D580 => array(0x75), 0x1D581 => array(0x76),
                    0x1D582 => array(0x77), 0x1D583 => array(0x78), 0x1D584 => array(0x79),
                    0x1D585 => array(0x7A), 0x1D586 => array(0x61), 0x1D587 => array(0x62),
                    0x1D588 => array(0x63), 0x1D589 => array(0x64), 0x1D58A => array(0x65),
                    0x1D58B => array(0x66), 0x1D58C => array(0x67), 0x1D58D => array(0x68),
                    0x1D58E => array(0x69), 0x1D58F => array(0x6A), 0x1D590 => array(0x6B),
                    0x1D591 => array(0x6C), 0x1D592 => array(0x6D), 0x1D593 => array(0x6E),
                    0x1D594 => array(0x6F), 0x1D595 => array(0x70), 0x1D596 => array(0x71),
                    0x1D597 => array(0x72), 0x1D598 => array(0x73), 0x1D599 => array(0x74),
                    0x1D59A => array(0x75), 0x1D59B => array(0x76), 0x1D59C => array(0x77),
                    0x1D59D => array(0x78), 0x1D59E => array(0x79), 0x1D59F => array(0x7A),
                    0x1D5A0 => array(0x61), 0x1D5A1 => array(0x62), 0x1D5A2 => array(0x63),
                    0x1D5A3 => array(0x64), 0x1D5A4 => array(0x65), 0x1D5A5 => array(0x66),
                    0x1D5A6 => array(0x67), 0x1D5A7 => array(0x68), 0x1D5A8 => array(0x69),
                    0x1D5A9 => array(0x6A), 0x1D5AA => array(0x6B), 0x1D5AB => array(0x6C),
                    0x1D5AC => array(0x6D), 0x1D5AD => array(0x6E), 0x1D5AE => array(0x6F),
                    0x1D5AF => array(0x70), 0x1D5B0 => array(0x71), 0x1D5B1 => array(0x72),
                    0x1D5B2 => array(0x73), 0x1D5B3 => array(0x74), 0x1D5B4 => array(0x75),
                    0x1D5B5 => array(0x76), 0x1D5B6 => array(0x77), 0x1D5B7 => array(0x78),
                    0x1D5B8 => array(0x79), 0x1D5B9 => array(0x7A), 0x1D5BA => array(0x61),
                    0x1D5BB => array(0x62), 0x1D5BC => array(0x63), 0x1D5BD => array(0x64),
                    0x1D5BE => array(0x65), 0x1D5BF => array(0x66), 0x1D5C0 => array(0x67),
                    0x1D5C1 => array(0x68), 0x1D5C2 => array(0x69), 0x1D5C3 => array(0x6A),
                    0x1D5C4 => array(0x6B), 0x1D5C5 => array(0x6C), 0x1D5C6 => array(0x6D),
                    0x1D5C7 => array(0x6E), 0x1D5C8 => array(0x6F), 0x1D5C9 => array(0x70),
                    0x1D5CA => array(0x71), 0x1D5CB => array(0x72), 0x1D5CC => array(0x73),
                    0x1D5CD => array(0x74), 0x1D5CE => array(0x75), 0x1D5CF => array(0x76),
                    0x1D5D0 => array(0x77), 0x1D5D1 => array(0x78), 0x1D5D2 => array(0x79),
                    0x1D5D3 => array(0x7A), 0x1D5D4 => array(0x61), 0x1D5D5 => array(0x62),
                    0x1D5D6 => array(0x63), 0x1D5D7 => array(0x64), 0x1D5D8 => array(0x65),
                    0x1D5D9 => array(0x66), 0x1D5DA => array(0x67), 0x1D5DB => array(0x68),
                    0x1D5DC => array(0x69), 0x1D5DD => array(0x6A), 0x1D5DE => array(0x6B),
                    0x1D5DF => array(0x6C), 0x1D5E0 => array(0x6D), 0x1D5E1 => array(0x6E),
                    0x1D5E2 => array(0x6F), 0x1D5E3 => array(0x70), 0x1D5E4 => array(0x71),
                    0x1D5E5 => array(0x72), 0x1D5E6 => array(0x73), 0x1D5E7 => array(0x74),
                    0x1D5E8 => array(0x75), 0x1D5E9 => array(0x76), 0x1D5EA => array(0x77),
                    0x1D5EB => array(0x78), 0x1D5EC => array(0x79), 0x1D5ED => array(0x7A),
                    0x1D5EE => array(0x61), 0x1D5EF => array(0x62), 0x1D5F0 => array(0x63),
                    0x1D5F1 => array(0x64), 0x1D5F2 => array(0x65), 0x1D5F3 => array(0x66),
                    0x1D5F4 => array(0x67), 0x1D5F5 => array(0x68), 0x1D5F6 => array(0x69),
                    0x1D5F7 => array(0x6A), 0x1D5F8 => array(0x6B), 0x1D5F9 => array(0x6C),
                    0x1D5FA => array(0x6D), 0x1D5FB => array(0x6E), 0x1D5FC => array(0x6F),
                    0x1D5FD => array(0x70), 0x1D5FE => array(0x71), 0x1D5FF => array(0x72),
                    0x1D600 => array(0x73), 0x1D601 => array(0x74), 0x1D602 => array(0x75),
                    0x1D603 => array(0x76), 0x1D604 => array(0x77), 0x1D605 => array(0x78),
                    0x1D606 => array(0x79), 0x1D607 => array(0x7A), 0x1D608 => array(0x61),
                    0x1D609 => array(0x62), 0x1D60A => array(0x63), 0x1D60B => array(0x64),
                    0x1D60C => array(0x65), 0x1D60D => array(0x66), 0x1D60E => array(0x67),
                    0x1D60F => array(0x68), 0x1D610 => array(0x69), 0x1D611 => array(0x6A),
                    0x1D612 => array(0x6B), 0x1D613 => array(0x6C), 0x1D614 => array(0x6D),
                    0x1D615 => array(0x6E), 0x1D616 => array(0x6F), 0x1D617 => array(0x70),
                    0x1D618 => array(0x71), 0x1D619 => array(0x72), 0x1D61A => array(0x73),
                    0x1D61B => array(0x74), 0x1D61C => array(0x75), 0x1D61D => array(0x76),
                    0x1D61E => array(0x77), 0x1D61F => array(0x78), 0x1D620 => array(0x79),
                    0x1D621 => array(0x7A), 0x1D622 => array(0x61), 0x1D623 => array(0x62),
                    0x1D624 => array(0x63), 0x1D625 => array(0x64), 0x1D626 => array(0x65),
                    0x1D627 => array(0x66), 0x1D628 => array(0x67), 0x1D629 => array(0x68),
                    0x1D62A => array(0x69), 0x1D62B => array(0x6A), 0x1D62C => array(0x6B),
                    0x1D62D => array(0x6C), 0x1D62E => array(0x6D), 0x1D62F => array(0x6E),
                    0x1D630 => array(0x6F), 0x1D631 => array(0x70), 0x1D632 => array(0x71),
                    0x1D633 => array(0x72), 0x1D634 => array(0x73), 0x1D635 => array(0x74),
                    0x1D636 => array(0x75), 0x1D637 => array(0x76), 0x1D638 => array(0x77),
                    0x1D639 => array(0x78), 0x1D63A => array(0x79), 0x1D63B => array(0x7A),
                    0x1D63C => array(0x61), 0x1D63D => array(0x62), 0x1D63E => array(0x63),
                    0x1D63F => array(0x64), 0x1D640 => array(0x65), 0x1D641 => array(0x66),
                    0x1D642 => array(0x67), 0x1D643 => array(0x68), 0x1D644 => array(0x69),
                    0x1D645 => array(0x6A), 0x1D646 => array(0x6B), 0x1D647 => array(0x6C),
                    0x1D648 => array(0x6D), 0x1D649 => array(0x6E), 0x1D64A => array(0x6F),
                    0x1D64B => array(0x70), 0x1D64C => array(0x71), 0x1D64D => array(0x72),
                    0x1D64E => array(0x73), 0x1D64F => array(0x74), 0x1D650 => array(0x75),
                    0x1D651 => array(0x76), 0x1D652 => array(0x77), 0x1D653 => array(0x78),
                    0x1D654 => array(0x79), 0x1D655 => array(0x7A), 0x1D656 => array(0x61),
                    0x1D657 => array(0x62), 0x1D658 => array(0x63), 0x1D659 => array(0x64),
                    0x1D65A => array(0x65), 0x1D65B => array(0x66), 0x1D65C => array(0x67),
                    0x1D65D => array(0x68), 0x1D65E => array(0x69), 0x1D65F => array(0x6A),
                    0x1D660 => array(0x6B), 0x1D661 => array(0x6C), 0x1D662 => array(0x6D),
                    0x1D663 => array(0x6E), 0x1D664 => array(0x6F), 0x1D665 => array(0x70),
                    0x1D666 => array(0x71), 0x1D667 => array(0x72), 0x1D668 => array(0x73),
                    0x1D669 => array(0x74), 0x1D66A => array(0x75), 0x1D66B => array(0x76),
                    0x1D66C => array(0x77), 0x1D66D => array(0x78), 0x1D66E => array(0x79),
                    0x1D66F => array(0x7A), 0x1D670 => array(0x61), 0x1D671 => array(0x62),
                    0x1D672 => array(0x63), 0x1D673 => array(0x64), 0x1D674 => array(0x65),
                    0x1D675 => array(0x66), 0x1D676 => array(0x67), 0x1D677 => array(0x68),
                    0x1D678 => array(0x69), 0x1D679 => array(0x6A), 0x1D67A => array(0x6B),
                    0x1D67B => array(0x6C), 0x1D67C => array(0x6D), 0x1D67D => array(0x6E),
                    0x1D67E => array(0x6F), 0x1D67F => array(0x70), 0x1D680 => array(0x71),
                    0x1D681 => array(0x72), 0x1D682 => array(0x73), 0x1D683 => array(0x74),
                    0x1D684 => array(0x75), 0x1D685 => array(0x76), 0x1D686 => array(0x77),
                    0x1D687 => array(0x78), 0x1D688 => array(0x79), 0x1D689 => array(0x7A),
                    0x1D68A => array(0x61), 0x1D68B => array(0x62), 0x1D68C => array(0x63),
                    0x1D68D => array(0x64), 0x1D68E => array(0x65), 0x1D68F => array(0x66),
                    0x1D690 => array(0x67), 0x1D691 => array(0x68), 0x1D692 => array(0x69),
                    0x1D693 => array(0x6A), 0x1D694 => array(0x6B), 0x1D695 => array(0x6C),
                    0x1D696 => array(0x6D), 0x1D697 => array(0x6E), 0x1D698 => array(0x6F),
                    0x1D699 => array(0x70), 0x1D69A => array(0x71), 0x1D69B => array(0x72),
                    0x1D69C => array(0x73), 0x1D69D => array(0x74), 0x1D69E => array(0x75),
                    0x1D69F => array(0x76), 0x1D6A0 => array(0x77), 0x1D6A1 => array(0x78),
                    0x1D6A2 => array(0x79), 0x1D6A3 => array(0x7A), 0x1D6A4 => array(0x131),
                    0x1D6A5 => array(0x237), 0x1D6A8 => array(0x3B1), 0x1D6A9 => array(0x3B2),
                    0x1D6AA => array(0x3B3), 0x1D6AB => array(0x3B4), 0x1D6AC => array(0x3B5),
                    0x1D6AD => array(0x3B6), 0x1D6AE => array(0x3B7), 0x1D6AF => array(0x3B8),
                    0x1D6B0 => array(0x3B9), 0x1D6B1 => array(0x3BA), 0x1D6B2 => array(0x3BB),
                    0x1D6B3 => array(0x3BC), 0x1D6B4 => array(0x3BD), 0x1D6B5 => array(0x3BE),
                    0x1D6B6 => array(0x3BF), 0x1D6B7 => array(0x3C0), 0x1D6B8 => array(0x3C1),
                    0x1D6B9 => array(0x3B8), 0x1D6BA => array(0x3C3), 0x1D6BB => array(0x3C4),
                    0x1D6BC => array(0x3C5), 0x1D6BD => array(0x3C6), 0x1D6BE => array(0x3C7),
                    0x1D6BF => array(0x3C8), 0x1D6C0 => array(0x3C9), 0x1D6C1 => array(0x2207),
                    0x1D6C2 => array(0x3B1), 0x1D6C3 => array(0x3B2), 0x1D6C4 => array(0x3B3),
                    0x1D6C5 => array(0x3B4), 0x1D6C6 => array(0x3B5), 0x1D6C7 => array(0x3B6),
                    0x1D6C8 => array(0x3B7), 0x1D6C9 => array(0x3B8), 0x1D6CA => array(0x3B9),
                    0x1D6CB => array(0x3BA), 0x1D6CC => array(0x3BB), 0x1D6CD => array(0x3BC),
                    0x1D6CE => array(0x3BD), 0x1D6CF => array(0x3BE), 0x1D6D0 => array(0x3BF),
                    0x1D6D1 => array(0x3C0), 0x1D6D2 => array(0x3C1), 0x1D6D3 => array(0x3C3),
                    0x1D6D4 => array(0x3C3), 0x1D6D5 => array(0x3C4), 0x1D6D6 => array(0x3C5),
                    0x1D6D7 => array(0x3C6), 0x1D6D8 => array(0x3C7), 0x1D6D9 => array(0x3C8),
                    0x1D6DA => array(0x3C9), 0x1D6DB => array(0x2202), 0x1D6DC => array(0x3B5),
                    0x1D6DD => array(0x3B8), 0x1D6DE => array(0x3BA), 0x1D6DF => array(0x3C6),
                    0x1D6E0 => array(0x3C1), 0x1D6E1 => array(0x3C0), 0x1D6E2 => array(0x3B1),
                    0x1D6E3 => array(0x3B2), 0x1D6E4 => array(0x3B3), 0x1D6E5 => array(0x3B4),
                    0x1D6E6 => array(0x3B5), 0x1D6E7 => array(0x3B6), 0x1D6E8 => array(0x3B7),
                    0x1D6E9 => array(0x3B8), 0x1D6EA => array(0x3B9), 0x1D6EB => array(0x3BA),
                    0x1D6EC => array(0x3BB), 0x1D6ED => array(0x3BC), 0x1D6EE => array(0x3BD),
                    0x1D6EF => array(0x3BE), 0x1D6F0 => array(0x3BF), 0x1D6F1 => array(0x3C0),
                    0x1D6F2 => array(0x3C1), 0x1D6F3 => array(0x3B8), 0x1D6F4 => array(0x3C3),
                    0x1D6F5 => array(0x3C4), 0x1D6F6 => array(0x3C5), 0x1D6F7 => array(0x3C6),
                    0x1D6F8 => array(0x3C7), 0x1D6F9 => array(0x3C8), 0x1D6FA => array(0x3C9),
                    0x1D6FB => array(0x2207), 0x1D6FC => array(0x3B1), 0x1D6FD => array(0x3B2),
                    0x1D6FE => array(0x3B3), 0x1D6FF => array(0x3B4), 0x1D700 => array(0x3B5),
                    0x1D701 => array(0x3B6), 0x1D702 => array(0x3B7), 0x1D703 => array(0x3B8),
                    0x1D704 => array(0x3B9), 0x1D705 => array(0x3BA), 0x1D706 => array(0x3BB),
                    0x1D707 => array(0x3BC), 0x1D708 => array(0x3BD), 0x1D709 => array(0x3BE),
                    0x1D70A => array(0x3BF), 0x1D70B => array(0x3C0), 0x1D70C => array(0x3C1),
                    0x1D70D => array(0x3C3), 0x1D70E => array(0x3C3), 0x1D70F => array(0x3C4),
                    0x1D710 => array(0x3C5), 0x1D711 => array(0x3C6), 0x1D712 => array(0x3C7),
                    0x1D713 => array(0x3C8), 0x1D714 => array(0x3C9), 0x1D715 => array(0x2202),
                    0x1D716 => array(0x3B5), 0x1D717 => array(0x3B8), 0x1D718 => array(0x3BA),
                    0x1D719 => array(0x3C6), 0x1D71A => array(0x3C1), 0x1D71B => array(0x3C0),
                    0x1D71C => array(0x3B1), 0x1D71D => array(0x3B2), 0x1D71E => array(0x3B3),
                    0x1D71F => array(0x3B4), 0x1D720 => array(0x3B5), 0x1D721 => array(0x3B6),
                    0x1D722 => array(0x3B7), 0x1D723 => array(0x3B8), 0x1D724 => array(0x3B9),
                    0x1D725 => array(0x3BA), 0x1D726 => array(0x3BB), 0x1D727 => array(0x3BC),
                    0x1D728 => array(0x3BD), 0x1D729 => array(0x3BE), 0x1D72A => array(0x3BF),
                    0x1D72B => array(0x3C0), 0x1D72C => array(0x3C1), 0x1D72D => array(0x3B8),
                    0x1D72E => array(0x3C3), 0x1D72F => array(0x3C4), 0x1D730 => array(0x3C5),
                    0x1D731 => array(0x3C6), 0x1D732 => array(0x3C7), 0x1D733 => array(0x3C8),
                    0x1D734 => array(0x3C9), 0x1D735 => array(0x2207), 0x1D736 => array(0x3B1),
                    0x1D737 => array(0x3B2), 0x1D738 => array(0x3B3), 0x1D739 => array(0x3B4),
                    0x1D73A => array(0x3B5), 0x1D73B => array(0x3B6), 0x1D73C => array(0x3B7),
                    0x1D73D => array(0x3B8), 0x1D73E => array(0x3B9), 0x1D73F => array(0x3BA),
                    0x1D740 => array(0x3BB), 0x1D741 => array(0x3BC), 0x1D742 => array(0x3BD),
                    0x1D743 => array(0x3BE), 0x1D744 => array(0x3BF), 0x1D745 => array(0x3C0),
                    0x1D746 => array(0x3C1), 0x1D747 => array(0x3C3), 0x1D748 => array(0x3C3),
                    0x1D749 => array(0x3C4), 0x1D74A => array(0x3C5), 0x1D74B => array(0x3C6),
                    0x1D74C => array(0x3C7), 0x1D74D => array(0x3C8), 0x1D74E => array(0x3C9),
                    0x1D74F => array(0x2202), 0x1D750 => array(0x3B5), 0x1D751 => array(0x3B8),
                    0x1D752 => array(0x3BA), 0x1D753 => array(0x3C6), 0x1D754 => array(0x3C1),
                    0x1D755 => array(0x3C0), 0x1D756 => array(0x3B1), 0x1D757 => array(0x3B2),
                    0x1D758 => array(0x3B3), 0x1D759 => array(0x3B4), 0x1D75A => array(0x3B5),
                    0x1D75B => array(0x3B6), 0x1D75C => array(0x3B7), 0x1D75D => array(0x3B8),
                    0x1D75E => array(0x3B9), 0x1D75F => array(0x3BA), 0x1D760 => array(0x3BB),
                    0x1D761 => array(0x3BC), 0x1D762 => array(0x3BD), 0x1D763 => array(0x3BE),
                    0x1D764 => array(0x3BF), 0x1D765 => array(0x3C0), 0x1D766 => array(0x3C1),
                    0x1D767 => array(0x3B8), 0x1D768 => array(0x3C3), 0x1D769 => array(0x3C4),
                    0x1D76A => array(0x3C5), 0x1D76B => array(0x3C6), 0x1D76C => array(0x3C7),
                    0x1D76D => array(0x3C8), 0x1D76E => array(0x3C9), 0x1D76F => array(0x2207),
                    0x1D770 => array(0x3B1), 0x1D771 => array(0x3B2), 0x1D772 => array(0x3B3),
                    0x1D773 => array(0x3B4), 0x1D774 => array(0x3B5), 0x1D775 => array(0x3B6),
                    0x1D776 => array(0x3B7), 0x1D777 => array(0x3B8), 0x1D778 => array(0x3B9),
                    0x1D779 => array(0x3BA), 0x1D77A => array(0x3BB), 0x1D77B => array(0x3BC),
                    0x1D77C => array(0x3BD), 0x1D77D => array(0x3BE), 0x1D77E => array(0x3BF),
                    0x1D77F => array(0x3C0), 0x1D780 => array(0x3C1), 0x1D781 => array(0x3C3),
                    0x1D782 => array(0x3C3), 0x1D783 => array(0x3C4), 0x1D784 => array(0x3C5),
                    0x1D785 => array(0x3C6), 0x1D786 => array(0x3C7), 0x1D787 => array(0x3C8),
                    0x1D788 => array(0x3C9), 0x1D789 => array(0x2202), 0x1D78A => array(0x3B5),
                    0x1D78B => array(0x3B8), 0x1D78C => array(0x3BA), 0x1D78D => array(0x3C6),
                    0x1D78E => array(0x3C1), 0x1D78F => array(0x3C0), 0x1D790 => array(0x3B1),
                    0x1D791 => array(0x3B2), 0x1D792 => array(0x3B3), 0x1D793 => array(0x3B4),
                    0x1D794 => array(0x3B5), 0x1D795 => array(0x3B6), 0x1D796 => array(0x3B7),
                    0x1D797 => array(0x3B8), 0x1D798 => array(0x3B9), 0x1D799 => array(0x3BA),
                    0x1D79A => array(0x3BB), 0x1D79B => array(0x3BC), 0x1D79C => array(0x3BD),
                    0x1D79D => array(0x3BE), 0x1D79E => array(0x3BF), 0x1D79F => array(0x3C0),
                    0x1D7A0 => array(0x3C1), 0x1D7A1 => array(0x3B8), 0x1D7A2 => array(0x3C3),
                    0x1D7A3 => array(0x3C4), 0x1D7A4 => array(0x3C5), 0x1D7A5 => array(0x3C6),
                    0x1D7A6 => array(0x3C7), 0x1D7A7 => array(0x3C8), 0x1D7A8 => array(0x3C9),
                    0x1D7A9 => array(0x2207), 0x1D7AA => array(0x3B1), 0x1D7AB => array(0x3B2),
                    0x1D7AC => array(0x3B3), 0x1D7AD => array(0x3B4), 0x1D7AE => array(0x3B5),
                    0x1D7AF => array(0x3B6), 0x1D7B0 => array(0x3B7), 0x1D7B1 => array(0x3B8),
                    0x1D7B2 => array(0x3B9), 0x1D7B3 => array(0x3BA), 0x1D7B4 => array(0x3BB),
                    0x1D7B5 => array(0x3BC), 0x1D7B6 => array(0x3BD), 0x1D7B7 => array(0x3BE),
                    0x1D7B8 => array(0x3BF), 0x1D7B9 => array(0x3C0), 0x1D7BA => array(0x3C1),
                    0x1D7BB => array(0x3C3), 0x1D7BC => array(0x3C3), 0x1D7BD => array(0x3C4),
                    0x1D7BE => array(0x3C5), 0x1D7BF => array(0x3C6), 0x1D7C0 => array(0x3C7),
                    0x1D7C1 => array(0x3C8), 0x1D7C2 => array(0x3C9), 0x1D7C3 => array(0x2202),
                    0x1D7C4 => array(0x3B5), 0x1D7C5 => array(0x3B8), 0x1D7C6 => array(0x3BA),
                    0x1D7C7 => array(0x3C6), 0x1D7C8 => array(0x3C1), 0x1D7C9 => array(0x3C0),
                    0x1D7CA => array(0x3DD), 0x1D7CB => array(0x3DD), 0x1D7CE => array(0x30),
                    0x1D7CF => array(0x31), 0x1D7D0 => array(0x32), 0x1D7D1 => array(0x33),
                    0x1D7D2 => array(0x34), 0x1D7D3 => array(0x35), 0x1D7D4 => array(0x36),
                    0x1D7D5 => array(0x37), 0x1D7D6 => array(0x38), 0x1D7D7 => array(0x39),
                    0x1D7D8 => array(0x30), 0x1D7D9 => array(0x31), 0x1D7DA => array(0x32),
                    0x1D7DB => array(0x33), 0x1D7DC => array(0x34), 0x1D7DD => array(0x35),
                    0x1D7DE => array(0x36), 0x1D7DF => array(0x37), 0x1D7E0 => array(0x38),
                    0x1D7E1 => array(0x39), 0x1D7E2 => array(0x30), 0x1D7E3 => array(0x31),
                    0x1D7E4 => array(0x32), 0x1D7E5 => array(0x33), 0x1D7E6 => array(0x34),
                    0x1D7E7 => array(0x35), 0x1D7E8 => array(0x36), 0x1D7E9 => array(0x37),
                    0x1D7EA => array(0x38), 0x1D7EB => array(0x39), 0x1D7EC => array(0x30),
                    0x1D7ED => array(0x31), 0x1D7EE => array(0x32), 0x1D7EF => array(0x33),
                    0x1D7F0 => array(0x34), 0x1D7F1 => array(0x35), 0x1D7F2 => array(0x36),
                    0x1D7F3 => array(0x37), 0x1D7F4 => array(0x38), 0x1D7F5 => array(0x39),
                    0x1D7F6 => array(0x30), 0x1D7F7 => array(0x31), 0x1D7F8 => array(0x32),
                    0x1D7F9 => array(0x33), 0x1D7FA => array(0x34), 0x1D7FB => array(0x35),
                    0x1D7FC => array(0x36), 0x1D7FD => array(0x37), 0x1D7FE => array(0x38),
                    0x1D7FF => array(0x39), 0x1EE00 => array(0x627), 0x1EE01 => array(0x628),
                    0x1EE02 => array(0x62C), 0x1EE03 => array(0x62F), 0x1EE05 => array(0x648),
                    0x1EE06 => array(0x632), 0x1EE07 => array(0x62D), 0x1EE08 => array(0x637),
                    0x1EE09 => array(0x64A), 0x1EE0A => array(0x643), 0x1EE0B => array(0x644),
                    0x1EE0C => array(0x645), 0x1EE0D => array(0x646), 0x1EE0E => array(0x633),
                    0x1EE0F => array(0x639), 0x1EE10 => array(0x641), 0x1EE11 => array(0x635),
                    0x1EE12 => array(0x642), 0x1EE13 => array(0x631), 0x1EE14 => array(0x634),
                    0x1EE15 => array(0x62A), 0x1EE16 => array(0x62B), 0x1EE17 => array(0x62E),
                    0x1EE18 => array(0x630), 0x1EE19 => array(0x636), 0x1EE1A => array(0x638),
                    0x1EE1B => array(0x63A), 0x1EE1C => array(0x66E), 0x1EE1D => array(0x6BA),
                    0x1EE1E => array(0x6A1), 0x1EE1F => array(0x66F), 0x1EE21 => array(0x628),
                    0x1EE22 => array(0x62C), 0x1EE24 => array(0x647), 0x1EE27 => array(0x62D),
                    0x1EE29 => array(0x64A), 0x1EE2A => array(0x643), 0x1EE2B => array(0x644),
                    0x1EE2C => array(0x645), 0x1EE2D => array(0x646), 0x1EE2E => array(0x633),
                    0x1EE2F => array(0x639), 0x1EE30 => array(0x641), 0x1EE31 => array(0x635),
                    0x1EE32 => array(0x642), 0x1EE34 => array(0x634), 0x1EE35 => array(0x62A),
                    0x1EE36 => array(0x62B), 0x1EE37 => array(0x62E), 0x1EE39 => array(0x636),
                    0x1EE3B => array(0x63A), 0x1EE42 => array(0x62C), 0x1EE47 => array(0x62D),
                    0x1EE49 => array(0x64A), 0x1EE4B => array(0x644), 0x1EE4D => array(0x646),
                    0x1EE4E => array(0x633), 0x1EE4F => array(0x639), 0x1EE51 => array(0x635),
                    0x1EE52 => array(0x642), 0x1EE54 => array(0x634), 0x1EE57 => array(0x62E),
                    0x1EE59 => array(0x636), 0x1EE5B => array(0x63A), 0x1EE5D => array(0x6BA),
                    0x1EE5F => array(0x66F), 0x1EE61 => array(0x628), 0x1EE62 => array(0x62C),
                    0x1EE64 => array(0x647), 0x1EE67 => array(0x62D), 0x1EE68 => array(0x637),
                    0x1EE69 => array(0x64A), 0x1EE6A => array(0x643), 0x1EE6C => array(0x645),
                    0x1EE6D => array(0x646), 0x1EE6E => array(0x633), 0x1EE6F => array(0x639),
                    0x1EE70 => array(0x641), 0x1EE71 => array(0x635), 0x1EE72 => array(0x642),
                    0x1EE74 => array(0x634), 0x1EE75 => array(0x62A), 0x1EE76 => array(0x62B),
                    0x1EE77 => array(0x62E), 0x1EE79 => array(0x636), 0x1EE7A => array(0x638),
                    0x1EE7B => array(0x63A), 0x1EE7C => array(0x66E), 0x1EE7E => array(0x6A1),
                    0x1EE80 => array(0x627), 0x1EE81 => array(0x628), 0x1EE82 => array(0x62C),
                    0x1EE83 => array(0x62F), 0x1EE84 => array(0x647), 0x1EE85 => array(0x648),
                    0x1EE86 => array(0x632), 0x1EE87 => array(0x62D), 0x1EE88 => array(0x637),
                    0x1EE89 => array(0x64A), 0x1EE8B => array(0x644), 0x1EE8C => array(0x645),
                    0x1EE8D => array(0x646), 0x1EE8E => array(0x633), 0x1EE8F => array(0x639),
                    0x1EE90 => array(0x641), 0x1EE91 => array(0x635), 0x1EE92 => array(0x642),
                    0x1EE93 => array(0x631), 0x1EE94 => array(0x634), 0x1EE95 => array(0x62A),
                    0x1EE96 => array(0x62B), 0x1EE97 => array(0x62E), 0x1EE98 => array(0x630),
                    0x1EE99 => array(0x636), 0x1EE9A => array(0x638), 0x1EE9B => array(0x63A),
                    0x1EEA1 => array(0x628), 0x1EEA2 => array(0x62C), 0x1EEA3 => array(0x62F),
                    0x1EEA5 => array(0x648), 0x1EEA6 => array(0x632), 0x1EEA7 => array(0x62D),
                    0x1EEA8 => array(0x637), 0x1EEA9 => array(0x64A), 0x1EEAB => array(0x644),
                    0x1EEAC => array(0x645), 0x1EEAD => array(0x646), 0x1EEAE => array(0x633),
                    0x1EEAF => array(0x639), 0x1EEB0 => array(0x641), 0x1EEB1 => array(0x635),
                    0x1EEB2 => array(0x642), 0x1EEB3 => array(0x631), 0x1EEB4 => array(0x634),
                    0x1EEB5 => array(0x62A), 0x1EEB6 => array(0x62B), 0x1EEB7 => array(0x62E),
                    0x1EEB8 => array(0x630), 0x1EEB9 => array(0x636), 0x1EEBA => array(0x638),
                    0x1EEBB => array(0x63A), 0x1F12A => array(0x3014, 0x73, 0x3015), 0x1F12B => array(0x63),
                    0x1F12C => array(0x72), 0x1F12D => array(0x63, 0x64), 0x1F12E => array(0x77, 0x7A),
                    0x1F130 => array(0x61), 0x1F131 => array(0x62), 0x1F132 => array(0x63),
                    0x1F133 => array(0x64), 0x1F134 => array(0x65), 0x1F135 => array(0x66),
                    0x1F136 => array(0x67), 0x1F137 => array(0x68), 0x1F138 => array(0x69),
                    0x1F139 => array(0x6A), 0x1F13A => array(0x6B), 0x1F13B => array(0x6C),
                    0x1F13C => array(0x6D), 0x1F13D => array(0x6E), 0x1F13E => array(0x6F),
                    0x1F13F => array(0x70), 0x1F140 => array(0x71), 0x1F141 => array(0x72),
                    0x1F142 => array(0x73), 0x1F143 => array(0x74), 0x1F144 => array(0x75),
                    0x1F145 => array(0x76), 0x1F146 => array(0x77), 0x1F147 => array(0x78),
                    0x1F148 => array(0x79), 0x1F149 => array(0x7A), 0x1F14A => array(0x68, 0x76),
                    0x1F14B => array(0x6D, 0x76), 0x1F14C => array(0x73, 0x64), 0x1F14D => array(0x73, 0x73),
                    0x1F14E => array(0x70, 0x70, 0x76), 0x1F14F => array(0x77, 0x63), 0x1F16A => array(0x6D, 0x63),
                    0x1F16B => array(0x6D, 0x64), 0x1F190 => array(0x64, 0x6A), 0x1F200 => array(0x307B, 0x304B),
                    0x1F201 => array(0x30B3, 0x30B3), 0x1F202 => array(0x30B5), 0x1F210 => array(0x624B),
                    0x1F211 => array(0x5B57), 0x1F212 => array(0x53CC), 0x1F213 => array(0x30C7),
                    0x1F214 => array(0x4E8C), 0x1F215 => array(0x591A), 0x1F216 => array(0x89E3),
                    0x1F217 => array(0x5929), 0x1F218 => array(0x4EA4), 0x1F219 => array(0x6620),
                    0x1F21A => array(0x7121), 0x1F21B => array(0x6599), 0x1F21C => array(0x524D),
                    0x1F21D => array(0x5F8C), 0x1F21E => array(0x518D), 0x1F21F => array(0x65B0),
                    0x1F220 => array(0x521D), 0x1F221 => array(0x7D42), 0x1F222 => array(0x751F),
                    0x1F223 => array(0x8CA9), 0x1F224 => array(0x58F0), 0x1F225 => array(0x5439),
                    0x1F226 => array(0x6F14), 0x1F227 => array(0x6295), 0x1F228 => array(0x6355),
                    0x1F229 => array(0x4E00), 0x1F22A => array(0x4E09), 0x1F22B => array(0x904A),
                    0x1F22C => array(0x5DE6), 0x1F22D => array(0x4E2D), 0x1F22E => array(0x53F3),
                    0x1F22F => array(0x6307), 0x1F230 => array(0x8D70), 0x1F231 => array(0x6253),
                    0x1F232 => array(0x7981), 0x1F233 => array(0x7A7A), 0x1F234 => array(0x5408),
                    0x1F235 => array(0x6E80), 0x1F236 => array(0x6709), 0x1F237 => array(0x6708),
                    0x1F238 => array(0x7533), 0x1F239 => array(0x5272), 0x1F23A => array(0x55B6),
                    0x1F240 => array(0x3014, 0x672C, 0x3015), 0x1F241 => array(0x3014, 0x4E09, 0x3015), 0x1F242 => array(0x3014, 0x4E8C, 0x3015),
                    0x1F243 => array(0x3014, 0x5B89, 0x3015), 0x1F244 => array(0x3014, 0x70B9, 0x3015), 0x1F245 => array(0x3014, 0x6253, 0x3015),
                    0x1F246 => array(0x3014, 0x76D7, 0x3015), 0x1F247 => array(0x3014, 0x52DD, 0x3015), 0x1F248 => array(0x3014, 0x6557, 0x3015),
                    0x1F250 => array(0x5F97), 0x1F251 => array(0x53EF), 0x2F800 => array(0x4E3D),
                    0x2F801 => array(0x4E38), 0x2F802 => array(0x4E41), 0x2F803 => array(0x20122),
                    0x2F804 => array(0x4F60), 0x2F805 => array(0x4FAE), 0x2F806 => array(0x4FBB),
                    0x2F807 => array(0x5002), 0x2F808 => array(0x507A), 0x2F809 => array(0x5099),
                    0x2F80A => array(0x50E7), 0x2F80B => array(0x50CF), 0x2F80C => array(0x349E),
                    0x2F80D => array(0x2063A), 0x2F80E => array(0x514D), 0x2F80F => array(0x5154),
                    0x2F810 => array(0x5164), 0x2F811 => array(0x5177), 0x2F812 => array(0x2051C),
                    0x2F813 => array(0x34B9), 0x2F814 => array(0x5167), 0x2F815 => array(0x518D),
                    0x2F816 => array(0x2054B), 0x2F817 => array(0x5197), 0x2F818 => array(0x51A4),
                    0x2F819 => array(0x4ECC), 0x2F81A => array(0x51AC), 0x2F81B => array(0x51B5),
                    0x2F81C => array(0x291DF), 0x2F81D => array(0x51F5), 0x2F81E => array(0x5203),
                    0x2F81F => array(0x34DF), 0x2F820 => array(0x523B), 0x2F821 => array(0x5246),
                    0x2F822 => array(0x5272), 0x2F823 => array(0x5277), 0x2F824 => array(0x3515),
                    0x2F825 => array(0x52C7), 0x2F826 => array(0x52C9), 0x2F827 => array(0x52E4),
                    0x2F828 => array(0x52FA), 0x2F829 => array(0x5305), 0x2F82A => array(0x5306),
                    0x2F82B => array(0x5317), 0x2F82C => array(0x5349), 0x2F82D => array(0x5351),
                    0x2F82E => array(0x535A), 0x2F82F => array(0x5373), 0x2F830 => array(0x537D),
                    0x2F831 => array(0x537F), 0x2F832 => array(0x537F), 0x2F833 => array(0x537F),
                    0x2F834 => array(0x20A2C), 0x2F835 => array(0x7070), 0x2F836 => array(0x53CA),
                    0x2F837 => array(0x53DF), 0x2F838 => array(0x20B63), 0x2F839 => array(0x53EB),
                    0x2F83A => array(0x53F1), 0x2F83B => array(0x5406), 0x2F83C => array(0x549E),
                    0x2F83D => array(0x5438), 0x2F83E => array(0x5448), 0x2F83F => array(0x5468),
                    0x2F840 => array(0x54A2), 0x2F841 => array(0x54F6), 0x2F842 => array(0x5510),
                    0x2F843 => array(0x5553), 0x2F844 => array(0x5563), 0x2F845 => array(0x5584),
                    0x2F846 => array(0x5584), 0x2F847 => array(0x5599), 0x2F848 => array(0x55AB),
                    0x2F849 => array(0x55B3), 0x2F84A => array(0x55C2), 0x2F84B => array(0x5716),
                    0x2F84C => array(0x5606), 0x2F84D => array(0x5717), 0x2F84E => array(0x5651),
                    0x2F84F => array(0x5674), 0x2F850 => array(0x5207), 0x2F851 => array(0x58EE),
                    0x2F852 => array(0x57CE), 0x2F853 => array(0x57F4), 0x2F854 => array(0x580D),
                    0x2F855 => array(0x578B), 0x2F856 => array(0x5832), 0x2F857 => array(0x5831),
                    0x2F858 => array(0x58AC), 0x2F859 => array(0x214E4), 0x2F85A => array(0x58F2),
                    0x2F85B => array(0x58F7), 0x2F85C => array(0x5906), 0x2F85D => array(0x591A),
                    0x2F85E => array(0x5922), 0x2F85F => array(0x5962), 0x2F860 => array(0x216A8),
                    0x2F861 => array(0x216EA), 0x2F862 => array(0x59EC), 0x2F863 => array(0x5A1B),
                    0x2F864 => array(0x5A27), 0x2F865 => array(0x59D8), 0x2F866 => array(0x5A66),
                    0x2F867 => array(0x36EE), 0x2F869 => array(0x5B08), 0x2F86A => array(0x5B3E),
                    0x2F86B => array(0x5B3E), 0x2F86C => array(0x219C8), 0x2F86D => array(0x5BC3),
                    0x2F86E => array(0x5BD8), 0x2F86F => array(0x5BE7), 0x2F870 => array(0x5BF3),
                    0x2F871 => array(0x21B18), 0x2F872 => array(0x5BFF), 0x2F873 => array(0x5C06),
                    0x2F875 => array(0x5C22), 0x2F876 => array(0x3781), 0x2F877 => array(0x5C60),
                    0x2F878 => array(0x5C6E), 0x2F879 => array(0x5CC0), 0x2F87A => array(0x5C8D),
                    0x2F87B => array(0x21DE4), 0x2F87C => array(0x5D43), 0x2F87D => array(0x21DE6),
                    0x2F87E => array(0x5D6E), 0x2F87F => array(0x5D6B), 0x2F880 => array(0x5D7C),
                    0x2F881 => array(0x5DE1), 0x2F882 => array(0x5DE2), 0x2F883 => array(0x382F),
                    0x2F884 => array(0x5DFD), 0x2F885 => array(0x5E28), 0x2F886 => array(0x5E3D),
                    0x2F887 => array(0x5E69), 0x2F888 => array(0x3862), 0x2F889 => array(0x22183),
                    0x2F88A => array(0x387C), 0x2F88B => array(0x5EB0), 0x2F88C => array(0x5EB3),
                    0x2F88D => array(0x5EB6), 0x2F88E => array(0x5ECA), 0x2F88F => array(0x2A392),
                    0x2F890 => array(0x5EFE), 0x2F891 => array(0x22331), 0x2F892 => array(0x22331),
                    0x2F893 => array(0x8201), 0x2F894 => array(0x5F22), 0x2F895 => array(0x5F22),
                    0x2F896 => array(0x38C7), 0x2F897 => array(0x232B8), 0x2F898 => array(0x261DA),
                    0x2F899 => array(0x5F62), 0x2F89A => array(0x5F6B), 0x2F89B => array(0x38E3),
                    0x2F89C => array(0x5F9A), 0x2F89D => array(0x5FCD), 0x2F89E => array(0x5FD7),
                    0x2F89F => array(0x5FF9), 0x2F8A0 => array(0x6081), 0x2F8A1 => array(0x393A),
                    0x2F8A2 => array(0x391C), 0x2F8A3 => array(0x6094), 0x2F8A4 => array(0x226D4),
                    0x2F8A5 => array(0x60C7), 0x2F8A6 => array(0x6148), 0x2F8A7 => array(0x614C),
                    0x2F8A8 => array(0x614E), 0x2F8A9 => array(0x614C), 0x2F8AA => array(0x617A),
                    0x2F8AB => array(0x618E), 0x2F8AC => array(0x61B2), 0x2F8AD => array(0x61A4),
                    0x2F8AE => array(0x61AF), 0x2F8AF => array(0x61DE), 0x2F8B0 => array(0x61F2),
                    0x2F8B1 => array(0x61F6), 0x2F8B2 => array(0x6210), 0x2F8B3 => array(0x621B),
                    0x2F8B4 => array(0x625D), 0x2F8B5 => array(0x62B1), 0x2F8B6 => array(0x62D4),
                    0x2F8B7 => array(0x6350), 0x2F8B8 => array(0x22B0C), 0x2F8B9 => array(0x633D),
                    0x2F8BA => array(0x62FC), 0x2F8BB => array(0x6368), 0x2F8BC => array(0x6383),
                    0x2F8BD => array(0x63E4), 0x2F8BE => array(0x22BF1), 0x2F8BF => array(0x6422),
                    0x2F8C0 => array(0x63C5), 0x2F8C1 => array(0x63A9), 0x2F8C2 => array(0x3A2E),
                    0x2F8C3 => array(0x6469), 0x2F8C4 => array(0x647E), 0x2F8C5 => array(0x649D),
                    0x2F8C6 => array(0x6477), 0x2F8C7 => array(0x3A6C), 0x2F8C8 => array(0x654F),
                    0x2F8C9 => array(0x656C), 0x2F8CA => array(0x2300A), 0x2F8CB => array(0x65E3),
                    0x2F8CC => array(0x66F8), 0x2F8CD => array(0x6649), 0x2F8CE => array(0x3B19),
                    0x2F8CF => array(0x6691), 0x2F8D0 => array(0x3B08), 0x2F8D1 => array(0x3AE4),
                    0x2F8D2 => array(0x5192), 0x2F8D3 => array(0x5195), 0x2F8D4 => array(0x6700),
                    0x2F8D5 => array(0x669C), 0x2F8D6 => array(0x80AD), 0x2F8D7 => array(0x43D9),
                    0x2F8D8 => array(0x6717), 0x2F8D9 => array(0x671B), 0x2F8DA => array(0x6721),
                    0x2F8DB => array(0x675E), 0x2F8DC => array(0x6753), 0x2F8DD => array(0x233C3),
                    0x2F8DE => array(0x3B49), 0x2F8DF => array(0x67FA), 0x2F8E0 => array(0x6785),
                    0x2F8E1 => array(0x6852), 0x2F8E2 => array(0x6885), 0x2F8E3 => array(0x2346D),
                    0x2F8E4 => array(0x688E), 0x2F8E5 => array(0x681F), 0x2F8E6 => array(0x6914),
                    0x2F8E7 => array(0x3B9D), 0x2F8E8 => array(0x6942), 0x2F8E9 => array(0x69A3),
                    0x2F8EA => array(0x69EA), 0x2F8EB => array(0x6AA8), 0x2F8EC => array(0x236A3),
                    0x2F8ED => array(0x6ADB), 0x2F8EE => array(0x3C18), 0x2F8EF => array(0x6B21),
                    0x2F8F0 => array(0x238A7), 0x2F8F1 => array(0x6B54), 0x2F8F2 => array(0x3C4E),
                    0x2F8F3 => array(0x6B72), 0x2F8F4 => array(0x6B9F), 0x2F8F5 => array(0x6BBA),
                    0x2F8F6 => array(0x6BBB), 0x2F8F7 => array(0x23A8D), 0x2F8F8 => array(0x21D0B),
                    0x2F8F9 => array(0x23AFA), 0x2F8FA => array(0x6C4E), 0x2F8FB => array(0x23CBC),
                    0x2F8FC => array(0x6CBF), 0x2F8FD => array(0x6CCD), 0x2F8FE => array(0x6C67),
                    0x2F8FF => array(0x6D16), 0x2F900 => array(0x6D3E), 0x2F901 => array(0x6D77),
                    0x2F902 => array(0x6D41), 0x2F903 => array(0x6D69), 0x2F904 => array(0x6D78),
                    0x2F905 => array(0x6D85), 0x2F906 => array(0x23D1E), 0x2F907 => array(0x6D34),
                    0x2F908 => array(0x6E2F), 0x2F909 => array(0x6E6E), 0x2F90A => array(0x3D33),
                    0x2F90B => array(0x6ECB), 0x2F90C => array(0x6EC7), 0x2F90D => array(0x23ED1),
                    0x2F90E => array(0x6DF9), 0x2F90F => array(0x6F6E), 0x2F910 => array(0x23F5E),
                    0x2F911 => array(0x23F8E), 0x2F912 => array(0x6FC6), 0x2F913 => array(0x7039),
                    0x2F914 => array(0x701E), 0x2F915 => array(0x701B), 0x2F916 => array(0x3D96),
                    0x2F917 => array(0x704A), 0x2F918 => array(0x707D), 0x2F919 => array(0x7077),
                    0x2F91A => array(0x70AD), 0x2F91B => array(0x20525), 0x2F91C => array(0x7145),
                    0x2F91D => array(0x24263), 0x2F91E => array(0x719C), 0x2F920 => array(0x7228),
                    0x2F921 => array(0x7235), 0x2F922 => array(0x7250), 0x2F923 => array(0x24608),
                    0x2F924 => array(0x7280), 0x2F925 => array(0x7295), 0x2F926 => array(0x24735),
                    0x2F927 => array(0x24814), 0x2F928 => array(0x737A), 0x2F929 => array(0x738B),
                    0x2F92A => array(0x3EAC), 0x2F92B => array(0x73A5), 0x2F92C => array(0x3EB8),
                    0x2F92D => array(0x3EB8), 0x2F92E => array(0x7447), 0x2F92F => array(0x745C),
                    0x2F930 => array(0x7471), 0x2F931 => array(0x7485), 0x2F932 => array(0x74CA),
                    0x2F933 => array(0x3F1B), 0x2F934 => array(0x7524), 0x2F935 => array(0x24C36),
                    0x2F936 => array(0x753E), 0x2F937 => array(0x24C92), 0x2F938 => array(0x7570),
                    0x2F939 => array(0x2219F), 0x2F93A => array(0x7610), 0x2F93B => array(0x24FA1),
                    0x2F93C => array(0x24FB8), 0x2F93D => array(0x25044), 0x2F93E => array(0x3FFC),
                    0x2F93F => array(0x4008), 0x2F940 => array(0x76F4), 0x2F941 => array(0x250F3),
                    0x2F942 => array(0x250F2), 0x2F943 => array(0x25119), 0x2F944 => array(0x25133),
                    0x2F945 => array(0x771E), 0x2F946 => array(0x771F), 0x2F947 => array(0x771F),
                    0x2F948 => array(0x774A), 0x2F949 => array(0x4039), 0x2F94A => array(0x778B),
                    0x2F94B => array(0x4046), 0x2F94C => array(0x4096), 0x2F94D => array(0x2541D),
                    0x2F94E => array(0x784E), 0x2F94F => array(0x788C), 0x2F950 => array(0x78CC),
                    0x2F951 => array(0x40E3), 0x2F952 => array(0x25626), 0x2F953 => array(0x7956),
                    0x2F954 => array(0x2569A), 0x2F955 => array(0x256C5), 0x2F956 => array(0x798F),
                    0x2F957 => array(0x79EB), 0x2F958 => array(0x412F), 0x2F959 => array(0x7A40),
                    0x2F95A => array(0x7A4A), 0x2F95B => array(0x7A4F), 0x2F95C => array(0x2597C),
                    0x2F95D => array(0x25AA7), 0x2F95E => array(0x25AA7), 0x2F960 => array(0x4202),
                    0x2F961 => array(0x25BAB), 0x2F962 => array(0x7BC6), 0x2F963 => array(0x7BC9),
                    0x2F964 => array(0x4227), 0x2F965 => array(0x25C80), 0x2F966 => array(0x7CD2),
                    0x2F967 => array(0x42A0), 0x2F968 => array(0x7CE8), 0x2F969 => array(0x7CE3),
                    0x2F96A => array(0x7D00), 0x2F96B => array(0x25F86), 0x2F96C => array(0x7D63),
                    0x2F96D => array(0x4301), 0x2F96E => array(0x7DC7), 0x2F96F => array(0x7E02),
                    0x2F970 => array(0x7E45), 0x2F971 => array(0x4334), 0x2F972 => array(0x26228),
                    0x2F973 => array(0x26247), 0x2F974 => array(0x4359), 0x2F975 => array(0x262D9),
                    0x2F976 => array(0x7F7A), 0x2F977 => array(0x2633E), 0x2F978 => array(0x7F95),
                    0x2F979 => array(0x7FFA), 0x2F97A => array(0x8005), 0x2F97B => array(0x264DA),
                    0x2F97C => array(0x26523), 0x2F97D => array(0x8060), 0x2F97E => array(0x265A8),
                    0x2F97F => array(0x8070), 0x2F980 => array(0x2335F), 0x2F981 => array(0x43D5),
                    0x2F982 => array(0x80B2), 0x2F983 => array(0x8103), 0x2F984 => array(0x440B),
                    0x2F985 => array(0x813E), 0x2F986 => array(0x5AB5), 0x2F987 => array(0x267A7),
                    0x2F988 => array(0x267B5), 0x2F989 => array(0x23393), 0x2F98A => array(0x2339C),
                    0x2F98B => array(0x8201), 0x2F98C => array(0x8204), 0x2F98D => array(0x8F9E),
                    0x2F98E => array(0x446B), 0x2F98F => array(0x8291), 0x2F990 => array(0x828B),
                    0x2F991 => array(0x829D), 0x2F992 => array(0x52B3), 0x2F993 => array(0x82B1),
                    0x2F994 => array(0x82B3), 0x2F995 => array(0x82BD), 0x2F996 => array(0x82E6),
                    0x2F997 => array(0x26B3C), 0x2F998 => array(0x82E5), 0x2F999 => array(0x831D),
                    0x2F99A => array(0x8363), 0x2F99B => array(0x83AD), 0x2F99C => array(0x8323),
                    0x2F99D => array(0x83BD), 0x2F99E => array(0x83E7), 0x2F99F => array(0x8457),
                    0x2F9A0 => array(0x8353), 0x2F9A1 => array(0x83CA), 0x2F9A2 => array(0x83CC),
                    0x2F9A3 => array(0x83DC), 0x2F9A4 => array(0x26C36), 0x2F9A5 => array(0x26D6B),
                    0x2F9A6 => array(0x26CD5), 0x2F9A7 => array(0x452B), 0x2F9A8 => array(0x84F1),
                    0x2F9A9 => array(0x84F3), 0x2F9AA => array(0x8516), 0x2F9AB => array(0x273CA),
                    0x2F9AC => array(0x8564), 0x2F9AD => array(0x26F2C), 0x2F9AE => array(0x455D),
                    0x2F9AF => array(0x4561), 0x2F9B0 => array(0x26FB1), 0x2F9B1 => array(0x270D2),
                    0x2F9B2 => array(0x456B), 0x2F9B3 => array(0x8650), 0x2F9B4 => array(0x865C),
                    0x2F9B5 => array(0x8667), 0x2F9B6 => array(0x8669), 0x2F9B7 => array(0x86A9),
                    0x2F9B8 => array(0x8688), 0x2F9B9 => array(0x870E), 0x2F9BA => array(0x86E2),
                    0x2F9BB => array(0x8779), 0x2F9BC => array(0x8728), 0x2F9BD => array(0x876B),
                    0x2F9BE => array(0x8786), 0x2F9C0 => array(0x87E1), 0x2F9C1 => array(0x8801),
                    0x2F9C2 => array(0x45F9), 0x2F9C3 => array(0x8860), 0x2F9C4 => array(0x8863),
                    0x2F9C5 => array(0x27667), 0x2F9C6 => array(0x88D7), 0x2F9C7 => array(0x88DE),
                    0x2F9C8 => array(0x4635), 0x2F9C9 => array(0x88FA), 0x2F9CA => array(0x34BB),
                    0x2F9CB => array(0x278AE), 0x2F9CC => array(0x27966), 0x2F9CD => array(0x46BE),
                    0x2F9CE => array(0x46C7), 0x2F9CF => array(0x8AA0), 0x2F9D0 => array(0x8AED),
                    0x2F9D1 => array(0x8B8A), 0x2F9D2 => array(0x8C55), 0x2F9D3 => array(0x27CA8),
                    0x2F9D4 => array(0x8CAB), 0x2F9D5 => array(0x8CC1), 0x2F9D6 => array(0x8D1B),
                    0x2F9D7 => array(0x8D77), 0x2F9D8 => array(0x27F2F), 0x2F9D9 => array(0x20804),
                    0x2F9DA => array(0x8DCB), 0x2F9DB => array(0x8DBC), 0x2F9DC => array(0x8DF0),
                    0x2F9DD => array(0x208DE), 0x2F9DE => array(0x8ED4), 0x2F9DF => array(0x8F38),
                    0x2F9E0 => array(0x285D2), 0x2F9E1 => array(0x285ED), 0x2F9E2 => array(0x9094),
                    0x2F9E3 => array(0x90F1), 0x2F9E4 => array(0x9111), 0x2F9E5 => array(0x2872E),
                    0x2F9E6 => array(0x911B), 0x2F9E7 => array(0x9238), 0x2F9E8 => array(0x92D7),
                    0x2F9E9 => array(0x92D8), 0x2F9EA => array(0x927C), 0x2F9EB => array(0x93F9),
                    0x2F9EC => array(0x9415), 0x2F9ED => array(0x28BFA), 0x2F9EE => array(0x958B),
                    0x2F9EF => array(0x4995), 0x2F9F0 => array(0x95B7), 0x2F9F1 => array(0x28D77),
                    0x2F9F2 => array(0x49E6), 0x2F9F3 => array(0x96C3), 0x2F9F4 => array(0x5DB2),
                    0x2F9F5 => array(0x9723), 0x2F9F6 => array(0x29145), 0x2F9F7 => array(0x2921A),
                    0x2F9F8 => array(0x4A6E), 0x2F9F9 => array(0x4A76), 0x2F9FA => array(0x97E0),
                    0x2F9FB => array(0x2940A), 0x2F9FC => array(0x4AB2), 0x2F9FD => array(0x29496),
                    0x2F9FE => array(0x980B), 0x2F9FF => array(0x980B), 0x2FA00 => array(0x9829),
                    0x2FA01 => array(0x295B6), 0x2FA02 => array(0x98E2), 0x2FA03 => array(0x4B33),
                    0x2FA04 => array(0x9929), 0x2FA05 => array(0x99A7), 0x2FA06 => array(0x99C2),
                    0x2FA07 => array(0x99FE), 0x2FA08 => array(0x4BCE), 0x2FA09 => array(0x29B30),
                    0x2FA0A => array(0x9B12), 0x2FA0B => array(0x9C40), 0x2FA0C => array(0x9CFD),
                    0x2FA0D => array(0x4CCE), 0x2FA0E => array(0x4CED), 0x2FA0F => array(0x9D67),
                    0x2FA10 => array(0x2A0CE), 0x2FA11 => array(0x4CF8), 0x2FA12 => array(0x2A105),
                    0x2FA13 => array(0x2A20E), 0x2FA14 => array(0x2A291), 0x2FA15 => array(0x9EBB),
                    0x2FA16 => array(0x4D56), 0x2FA17 => array(0x9EF9), 0x2FA18 => array(0x9EFE),
                    0x2FA19 => array(0x9F05), 0x2FA1A => array(0x9F0F), 0x2FA1B => array(0x9F16),
                    0x2FA1C => array(0x9F3B), 0x2FA1D => array(0x2A600)
            ),
            'norm_combcls' => array(0x334 => 1, 0x335 => 1, 0x336 => 1, 0x337 => 1,
                    0x338 => 1, 0x93C => 7, 0x9BC => 7, 0xA3C => 7, 0xABC => 7,
                    0xB3C => 7, 0xCBC => 7, 0x1037 => 7, 0x3099 => 8, 0x309A => 8,
                    0x94D => 9, 0x9CD => 9, 0xA4D => 9, 0xACD => 9, 0xB4D => 9,
                    0xBCD => 9, 0xC4D => 9, 0xCCD => 9, 0xD4D => 9, 0xDCA => 9,
                    0xE3A => 9, 0xF84 => 9, 0x1039 => 9, 0x1714 => 9, 0x1734 => 9,
                    0x17D2 => 9, 0x5B0 => 10, 0x5B1 => 11, 0x5B2 => 12, 0x5B3 => 13,
                    0x5B4 => 14, 0x5B5 => 15, 0x5B6 => 16, 0x5B7 => 17, 0x5B8 => 18,
                    0x5B9 => 19, 0x5BB => 20, 0x5Bc => 21, 0x5BD => 22, 0x5BF => 23,
                    0x5C1 => 24, 0x5C2 => 25, 0xFB1E => 26, 0x64B => 27, 0x64C => 28,
                    0x64D => 29, 0x64E => 30, 0x64F => 31, 0x650 => 32, 0x651 => 33,
                    0x652 => 34, 0x670 => 35, 0x711 => 36, 0xC55 => 84, 0xC56 => 91,
                    0xE38 => 103, 0xE39 => 103, 0xE48 => 107, 0xE49 => 107, 0xE4A => 107,
                    0xE4B => 107, 0xEB8 => 118, 0xEB9 => 118, 0xEC8 => 122, 0xEC9 => 122,
                    0xECA => 122, 0xECB => 122, 0xF71 => 129, 0xF72 => 130, 0xF7A => 130,
                    0xF7B => 130, 0xF7C => 130, 0xF7D => 130, 0xF80 => 130, 0xF74 => 132,
                    0x321 => 202, 0x322 => 202, 0x327 => 202, 0x328 => 202, 0x31B => 216,
                    0xF39 => 216, 0x1D165 => 216, 0x1D166 => 216, 0x1D16E => 216, 0x1D16F => 216,
                    0x1D170 => 216, 0x1D171 => 216, 0x1D172 => 216, 0x302A => 218, 0x316 => 220,
                    0x317 => 220, 0x318 => 220, 0x319 => 220, 0x31C => 220, 0x31D => 220,
                    0x31E => 220, 0x31F => 220, 0x320 => 220, 0x323 => 220, 0x324 => 220,
                    0x325 => 220, 0x326 => 220, 0x329 => 220, 0x32A => 220, 0x32B => 220,
                    0x32C => 220, 0x32D => 220, 0x32E => 220, 0x32F => 220, 0x330 => 220,
                    0x331 => 220, 0x332 => 220, 0x333 => 220, 0x339 => 220, 0x33A => 220,
                    0x33B => 220, 0x33C => 220, 0x347 => 220, 0x348 => 220, 0x349 => 220,
                    0x34D => 220, 0x34E => 220, 0x353 => 220, 0x354 => 220, 0x355 => 220,
                    0x356 => 220, 0x591 => 220, 0x596 => 220, 0x59B => 220, 0x5A3 => 220,
                    0x5A4 => 220, 0x5A5 => 220, 0x5A6 => 220, 0x5A7 => 220, 0x5AA => 220,
                    0x655 => 220, 0x656 => 220, 0x6E3 => 220, 0x6EA => 220, 0x6ED => 220,
                    0x731 => 220, 0x734 => 220, 0x737 => 220, 0x738 => 220, 0x739 => 220,
                    0x73B => 220, 0x73C => 220, 0x73E => 220, 0x742 => 220, 0x744 => 220,
                    0x746 => 220, 0x748 => 220, 0x952 => 220, 0xF18 => 220, 0xF19 => 220,
                    0xF35 => 220, 0xF37 => 220, 0xFC6 => 220, 0x193B => 220, 0x20E8 => 220,
                    0x1D17B => 220, 0x1D17C => 220, 0x1D17D => 220, 0x1D17E => 220, 0x1D17F => 220,
                    0x1D180 => 220, 0x1D181 => 220, 0x1D182 => 220, 0x1D18A => 220, 0x1D18B => 220,
                    0x59A => 222, 0x5AD => 222, 0x1929 => 222, 0x302D => 222, 0x302E => 224,
                    0x302F => 224, 0x1D16D => 226, 0x5AE => 228, 0x18A9 => 228, 0x302B => 228,
                    0x300 => 230, 0x301 => 230, 0x302 => 230, 0x303 => 230, 0x304 => 230,
                    0x305 => 230, 0x306 => 230, 0x307 => 230, 0x308 => 230, 0x309 => 230,
                    0x30A => 230, 0x30B => 230, 0x30C => 230, 0x30D => 230, 0x30E => 230,
                    0x30F => 230, 0x310 => 230, 0x311 => 230, 0x312 => 230, 0x313 => 230,
                    0x314 => 230, 0x33D => 230, 0x33E => 230, 0x33F => 230, 0x340 => 230,
                    0x341 => 230, 0x342 => 230, 0x343 => 230, 0x344 => 230, 0x346 => 230,
                    0x34A => 230, 0x34B => 230, 0x34C => 230, 0x350 => 230, 0x351 => 230,
                    0x352 => 230, 0x357 => 230, 0x363 => 230, 0x364 => 230, 0x365 => 230,
                    0x366 => 230, 0x367 => 230, 0x368 => 230, 0x369 => 230, 0x36A => 230,
                    0x36B => 230, 0x36C => 230, 0x36D => 230, 0x36E => 230, 0x36F => 230,
                    0x483 => 230, 0x484 => 230, 0x485 => 230, 0x486 => 230, 0x592 => 230,
                    0x593 => 230, 0x594 => 230, 0x595 => 230, 0x597 => 230, 0x598 => 230,
                    0x599 => 230, 0x59C => 230, 0x59D => 230, 0x59E => 230, 0x59F => 230,
                    0x5A0 => 230, 0x5A1 => 230, 0x5A8 => 230, 0x5A9 => 230, 0x5AB => 230,
                    0x5AC => 230, 0x5AF => 230, 0x5C4 => 230, 0x610 => 230, 0x611 => 230,
                    0x612 => 230, 0x613 => 230, 0x614 => 230, 0x615 => 230, 0x653 => 230,
                    0x654 => 230, 0x657 => 230, 0x658 => 230, 0x6D6 => 230, 0x6D7 => 230,
                    0x6D8 => 230, 0x6D9 => 230, 0x6DA => 230, 0x6DB => 230, 0x6DC => 230,
                    0x6DF => 230, 0x6E0 => 230, 0x6E1 => 230, 0x6E2 => 230, 0x6E4 => 230,
                    0x6E7 => 230, 0x6E8 => 230, 0x6EB => 230, 0x6EC => 230, 0x730 => 230,
                    0x732 => 230, 0x733 => 230, 0x735 => 230, 0x736 => 230, 0x73A => 230,
                    0x73D => 230, 0x73F => 230, 0x740 => 230, 0x741 => 230, 0x743 => 230,
                    0x745 => 230, 0x747 => 230, 0x749 => 230, 0x74A => 230, 0x951 => 230,
                    0x953 => 230, 0x954 => 230, 0xF82 => 230, 0xF83 => 230, 0xF86 => 230,
                    0xF87 => 230, 0x170D => 230, 0x193A => 230, 0x20D0 => 230, 0x20D1 => 230,
                    0x20D4 => 230, 0x20D5 => 230, 0x20D6 => 230, 0x20D7 => 230, 0x20DB => 230,
                    0x20DC => 230, 0x20E1 => 230, 0x20E7 => 230, 0x20E9 => 230, 0xFE20 => 230,
                    0xFE21 => 230, 0xFE22 => 230, 0xFE23 => 230, 0x1D185 => 230, 0x1D186 => 230,
                    0x1D187 => 230, 0x1D189 => 230, 0x1D188 => 230, 0x1D1AA => 230, 0x1D1AB => 230,
                    0x1D1AC => 230, 0x1D1AD => 230, 0x315 => 232, 0x31A => 232, 0x302C => 232,
                    0x35F => 233, 0x362 => 233, 0x35D => 234, 0x35E => 234, 0x360 => 234,
                    0x361 => 234, 0x345 => 240
            )
    );
}
